# rcsid {$Id: picio.tcl,v 4.19 1996/05/21 16:15:04 mangin Rel $}
# This package is free software. Redistribution and use of this file
# are permitted without restrictions.
#
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
#	Frank Mangin
#	Frank.Mangin@ceram.fr  -  Frank.Mangin@sophia.inria.fr

set ioState(drawingFile) [pwd]
set ioState(loadFile)    [pwd]
set ioState(lastImlist)  {}
set ioState(lastSaved)   {{} {} {} {}}

proc setDrawingName {name} {

  .frame1.name configure -text $name
  if {[string length $name] < 10} {
    .frame1.name configure -width 10
  } else {
    .frame1.name configure -width [string length $name]
  }
  update
}

proc SaveDrawing {} {
  global ioState fsbox
  
  if {! ([isafile "$ioState(drawingFile)"] && [file writable "$ioState(drawingFile)"])} {
    return [SaveDrawingAs]
  } else {
    SaveDrawingInFile "$ioState(drawingFile)"
  }
}
    
proc SaveDrawingAs {} {
  global ioState fsbox
  
  set path [fsbox 1 {Save drawing in file:} \
  "$ioState(drawingFile)" {Save}]
  if {"$path" == ""} { return }  

  focus .
  if {[file exists "$path"]} {
    set status [ConfirmBox "File exists. Save anyway ?" {save} {} {abort} {}]
    if {"$status" == "save"} {
      global ioState fsbox
      set ioState(drawingFile) "$path"
      setDrawingName "$fsbox(basename)"
      SaveDrawingInFile "$ioState(drawingFile)"
    }    
  } else {
    global ioState fsbox
    set ioState(drawingFile) "$path"
    setDrawingName "$fsbox(basename)"
    SaveDrawingInFile "$ioState(drawingFile)"
  }
}

##
## Write the drawing images to auxiliary files
##   $path.<image name>
##

proc saveDrawingImages { path } {
  set new 0; set files {}
  foreach name [getImnames] {
    set file "$path.$name"
    if {! [file exists "$file"]} {
      set new 1
    }
    switch -exact -- [image type $name] {
      "photo" {
        if {[catch {$name write "$file"} err]} {
          MessageBox "Couldn't save \"$name\" image file:\n  $err"
        } else {
          lappend files $file
        }
      }
      "bitmap" {
        if {[catch {
          global TmpDir
          set stin [open [file join "$TmpDir" "picasso.$name.xbm"]]
          set data [read $stin]; close $stin
          set stout [open "$file" w]
          puts $stout "$data"; close $stout
        } err]} {
          MessageBox "Couldn't save \"$name\" image file:\n $err"
        } else {
          lappend files $file
        }
      }
    }
  }
  
  set list [join $files "\n"]
  if ($new) {
    MessageBox "WARNING! The image files listed below \n\
    have been saved along with this drawing.\n\
    They should always remains in the same directory \n\
    as the main drawing file!\n\n\
    $list\n"
  }
  return $files
}

##
##  Makes the real job of writing the drawing content to file
##

proc SaveDrawingInFile { path } {
  global ioState env
  
  waitCursor
  
  if {[file exists "$path"] &&
    (! [file writable "$path"])} {
      warn "Couldn't open $file for writing"
      restoreCursor
      return
    }
  
  set imFiles [saveDrawingImages "$path"]
  
  if {[catch {open "$path" w} handle]} {
    warn "Open failed : $handle"
    restoreCursor
    return
  }
  
  #
  #    Write header
  #
  
  global PicassoVersion
  set header "##   Created by picasso version $PicassoVersion   ##"
  puts $handle "##\n$header\n##\n\n"
  if {[info exists $env(USER)]} {
    puts $handle "##  User : $env(USER)"
  }
  puts $handle "##  Date : [clock format [clock seconds]]\n\n"
  puts $handle "##"
  puts $handle "## These image files were save along with this drawing,
##   they should always remain in the same directory as this file:"
  foreach f $imFiles {
    set base [file tail "$f"]
    puts $handle "##     $base"
  }
  puts $handle "##\n\n"
  .f2.f22.cv dtag all current
  
  #
  #    Write tcl commands 
  #
  
  puts $handle "###############################"
  puts $handle "####  Start of TCL source  ####"
  puts $handle "###############################\n"
  
  #  Global variables  #
  puts $handle "global loadVersion"
  if {[lindex $PicassoVersion 0] == ""} {
    puts $handle "set loadVersion Exp"
  } else {
    puts $handle "set loadVersion [lindex $PicassoVersion 0]\n"
  }

  ## Save image creation instructions
  ## Don't harcode image names to avoid image name clashes on reload
  
  puts $handle "##\n##  Image objects creation\n##
  "
  foreach im [getImnames] {
    set file [file join [file tail "$path"]]
    append file ".$im"
    set switches "-file \"$file\""
    foreach spec [$im configure] {
      lassign $spec opt foo foo foo val
      if {$opt != "-file" && "$val" != ""} {
        append switches " $opt $val"
      }
    }

    puts $handle "
if {\[catch {image create [image type $im] $switches} tmpNames($im)\]} {
 MessageBox \"Couldn't create $im image:
      
\$tmpNames($im)

Make sure you kept the auxiliary file
$file
in the same directory as the main drawing file\"
}"
  }
  
  puts $handle "##\n\n"
  
  ## Save canvas items
  foreach item [.f2.f22.cv find withtag all] {
    set tags [.f2.f22.cv gettags $item]
    
    #  Don't save control items  #
    if {[lmember $tags "ctl"]} {
      continue
    }
    
    #  Just take group tags  #
    set stags {}
    foreach tag $tags {
      if {[string match {tGr*} $tag] ||
      [string match {Gr*} $tag]} {
        lappend stags $tag
      }
    }
    lappend stags inCreation
    
    set options [getSwitches $item]
    lappend options {-tags} $stags
    
    # Don't harcode image names
    if {[.f2.f22.cv type $item] == "image"} {
      set i [expr "1+[lsearch -exact $options -image]"]
      set name [lindex $options $i]
      set options "[lrange $options 0 [expr {$i - 1}]] \"\$tmpNames($name)\" [lrange $options [incr i] end]"
    }
          
    puts $handle "if {\[catch {.f2.f22.cv create [.f2.f22.cv type $item] [.f2.f22.cv coords $item] $options} err\]} {
      MessageBox \$err
    }"
  }

  puts $handle "return 0"
  close $handle

  set ioState(lastSaved) [getCurrentState]
  restoreCursor
}

################################################################
####			Loading                             ####
################################################################

#
#   Interactive Load procs
#

proc IncludeDrawing {} {
  global ioState

  set path [fsbox 0 {Include drawing from file:} \
  "$ioState(loadFile)" {Include}]
  focus .
  if {"$path" != ""} {
    global ioState
    set ioState(loadFile) "$path"
    ReadDrawing "$path"
  }
}

proc LoadDrawing {} {
  global ioState

  set cstate [getCurrentState]
  if {($ioState(lastSaved) != $cstate) &&
    ($cstate != {{} {} {} {}})} {
      set status [ConfirmBox "Save current drawing ?" \
		  {save} "" {discard} ""]
      update
      if {"$status" == "save"} {
	SaveDrawing
      }
    }

  set path [fsbox 0 {Load drawing from file:} "$ioState(loadFile)" {Load}]
  focus .
  global fsbox
  if {$fsbox(action) == "ok"} {
    setDrawingName "$fsbox(basename)"
    LoadDrawingFromFile "$path"
   }
} 

proc LoadDrawingFromFile {path} {
  global ioState

  set ioState(loadFile) "$path"
  set ioState(drawingFile) "$path"
  
  .f2.f22.cv delete all
  ReadDrawing "$path"
}

##
##  ReadDrawing : makes the real job
##  of item creation and back-compatibility stuff
##

proc ReadDrawing {path} {

  waitCursor
  
  if {![file readable "$path"]} {
    warn "Couldn't open $path for reading"
    restoreCursor
    return
  }

  #  old versions don't set inCreation tag on  #
  #  loaded objects, so have to remember existing objects  #
  set oldIds [.f2.f22.cv find withtag all]
  
  global loadVersion
  set loadVersion ""

  set savedwd [pwd]
  if {[catch {cd [file dirname "$path"]} err]} {
    warn "Couldn't chdir to \"$path\""
    cd $savedwd
    restoreCursor
    return
  }
  if {[catch {source "$path"} err]} {
    MessageBox "The following error occured while loading\n\"$path\":\n $err"
  }
  cd $savedwd
  
  if {![regexp {[0-9]*[.][0-9]*[a-z]} "$loadVersion"]} {
    #  Try to get version number from header  #
    set loadVersion ""
    set st [open "$path" r]
    for {set i 0} {$i < 5} {incr i} {
      if {[gets $st line] < 0} {
	break
      }
      if {[regexp {^[ ]*[#]*[a-zA-Z ]*[vV]ersion[ ]+([0-9]*[.][0-9]*)} \
	   $line foo loadVersion]} {
	     break
	   }
    }
    close $st

    if {$loadVersion == ""} {
      warn "No version info found"
      set loadVersion 0
    }
  }

  ####
  ####  Back Compatibility stuff
  ####

  #  remove trailing letter from version string  #
  regexp {[0-9]*[.][0-9]} $loadVersion loadVersion
  
  if {$loadVersion <= 3.4} {
    # image anchor changed to center #
    foreach id [.f2.f22.cv find withtag all] {
      if {[.f2.f22.cv type $id] == "image"} {
	set w [lindex [.f2.f22.cv itemconfigure $id -width] 4]
	set h [lindex [.f2.f22.cv itemconfigure $id -height] 4]
	.f2.f22.cv move $id [expr $w/2] [expr $h/2]
      }
    }
  }

  #  add inCreation tags if necessary  #
  if {($loadVersion <= 3.6) } {
    .f2.f22.cv dtag all inCreation
    foreach id [.f2.f22.cv find withtag all] {
      if {[lsearch $oldIds $id] < 0} {
	.f2.f22.cv addtag inCreation withtag $id
	# there was a bug with toRotate tags up to 3.6 #
	.f2.f22.cv dtag $id toRotate
      }
    }
  }
  
  set crids [.f2.f22.cv find withtag inCreation]
  
  # Replace obsolete groupList and topGroup tags #
  if {($loadVersion <= 3.7) } {
    foreach id [.f2.f22.cv find withtag inCreation] {
      set tags [.f2.f22.cv gettags $id]
      regsub {topGroup} $tags {tGr} tags
      regsub {groupList} $tags {Gr} tags
      .f2.f22.cv itemconfigure $id -tags $tags
    }
  }    
  
  #  update group tags of newly created objects  #
  updateGroupTags inCreation

  .f2.f22.cv dtag all inCreation
  restoreCursor
}

################################################################
####		   Writing Postscript                       ####
################################################################

set ioState(psFile) [pwd]

proc buildWriteDialog { fmt writeCmd } {

  toplevel .psDg
  
  # Window manager configurations
  global tkVersion
  wm positionfrom .psDg ""
  wm sizefrom .psDg ""
  wm maxsize .psDg [winfo screenwidth .] [winfo screenheight .]
  wm minsize .psDg 10 10
  wm title .psDg "Picasso Write $fmt"

  grab set .psDg
  bindtags .psDg .psDg
  
  ####  Label  ####

  label .psDg.titleLb \
    -text {Postscript Writing Options}

  ####  Path  ####

  frame .psDg.pathFr \
    -borderwidth 0

  label .psDg.pathFr.lb \
    -text {Write to file:} \
    -relief flat

  entry .psDg.pathFr.pathEn \
    -relief sunken \
    -width 40

  button .psDg.pathFr.fsbBt \
    -relief raised \
    -text {File Box} \
    -command "
      grab release .psDg
      set path \[fsbox 1 {$fmt File Name:} \[.psDg.pathFr.pathEn get\]\]
      if {\"\$path\" != \"\"} {
        .psDg.pathFr.pathEn delete 0 end
        .psDg.pathFr.pathEn insert 0 \"\$path\"
      }
      grab set .psDg
      focus ."
  
  #  packing  #

  pack append .psDg.pathFr \
    .psDg.pathFr.lb {left padx 12} \
    .psDg.pathFr.pathEn {left} \
    .psDg.pathFr.fsbBt {left padx 12}

  ####  Dimensions  ####

  frame .psDg.dimFr -borderwidth 0

  label .psDg.dimFr.dimLb \
    -text {Region of drawing to be written:} \
    -relief flat
  
  frame .psDg.dimFr.butFr -borderwidth 0
  button .psDg.dimFr.butFr.wholeBt \
    -text {whole drawing} \
    -relief raised \
    -command {
      lassign [.f2.f22.cv bbox all] x1 y1 x2 y2
      .psDg.dimFr.parFr.entryFr.x1En delete 0 end
      .psDg.dimFr.parFr.entryFr.x1En insert 0 $x1
      .psDg.dimFr.parFr.entryFr.y1En delete 0 end
      .psDg.dimFr.parFr.entryFr.y1En insert 0 $y1
      .psDg.dimFr.parFr.entryFr.widthEn delete 0 end
      .psDg.dimFr.parFr.entryFr.widthEn insert 0 [expr $x2-$x1]
      .psDg.dimFr.parFr.entryFr.heightEn delete 0 end
      .psDg.dimFr.parFr.entryFr.heightEn insert 0 [expr $y2-$y1]
    }

  button .psDg.dimFr.butFr.selBt \
    -text {select region} \
    -relief raised \
    -command {
      set RegionDone 0
      bind .psDg <ButtonPress-1>   {iregionStart %X %Y}
      bind .psDg <ButtonRelease-1> {iregionEnd}
      bind .psDg <Button1-Motion>  {iregion_move %X %Y}
      wm withdraw .psDg
      saveMsg
      msg "Drag region with mouse button 1"
      tkwait var RegionDone
      bind .psDg <ButtonPress-1> {}
      bind .psDg <ButtonRelease-1> {}
      bind .psDg <Button1-Motion> {}
      .psDg.dimFr.parFr.entryFr.x1En delete 0 end
      .psDg.dimFr.parFr.entryFr.x1En insert 0 $RegionX1
      .psDg.dimFr.parFr.entryFr.y1En delete 0 end
      .psDg.dimFr.parFr.entryFr.y1En insert 0 $RegionY1
      .psDg.dimFr.parFr.entryFr.widthEn delete 0 end
      .psDg.dimFr.parFr.entryFr.widthEn insert 0 $RegionWidth
      .psDg.dimFr.parFr.entryFr.heightEn delete 0 end
      .psDg.dimFr.parFr.entryFr.heightEn insert 0 $RegionHeight
      restoreMsg
      wm deiconify .psDg
    }

  pack .psDg.dimFr.butFr.wholeBt \
    -side left -padx 20 -ipadx 6 -ipady 4
  pack .psDg.dimFr.butFr.selBt \
    -side left -padx 20 -ipadx 6 -ipady 4
  
  label .psDg.dimFr.parFr -borderwidth 0
  
  frame .psDg.dimFr.parFr.labelFr -borderwidth 0
  label .psDg.dimFr.parFr.labelFr.x1Lb \
    -text {upper left x:} -relief flat
  label .psDg.dimFr.parFr.labelFr.y1Lb \
    -text {upper left y:} -relief flat
  label .psDg.dimFr.parFr.labelFr.widthLb \
    -text {width:} -relief flat
  label .psDg.dimFr.parFr.labelFr.heightLb \
    -text {height:} -relief flat

  pack append .psDg.dimFr.parFr.labelFr \
    .psDg.dimFr.parFr.labelFr.x1Lb    {top fill} \
    .psDg.dimFr.parFr.labelFr.y1Lb    {top fill} \
    .psDg.dimFr.parFr.labelFr.widthLb {top fill} \
    .psDg.dimFr.parFr.labelFr.heightLb {top fill}

  frame .psDg.dimFr.parFr.entryFr -borderwidth 0
  entry .psDg.dimFr.parFr.entryFr.x1En \
    -relief sunken -width 10
  entry .psDg.dimFr.parFr.entryFr.y1En \
    -relief sunken -width 10
  entry .psDg.dimFr.parFr.entryFr.widthEn \
    -relief sunken -width 10
  entry .psDg.dimFr.parFr.entryFr.heightEn \
    -relief sunken -width 10
  pack append .psDg.dimFr.parFr.entryFr \
    .psDg.dimFr.parFr.entryFr.x1En    {top fill} \
    .psDg.dimFr.parFr.entryFr.y1En {top fill} \
    .psDg.dimFr.parFr.entryFr.widthEn {top fill} \
    .psDg.dimFr.parFr.entryFr.heightEn {top fill}

  #  packing  #

  pack append .psDg.dimFr.parFr \
    .psDg.dimFr.parFr.labelFr {left padx 20 frame e} \
    .psDg.dimFr.parFr.entryFr {left padx 8 frame w}

  pack append .psDg.dimFr \
    .psDg.dimFr.dimLb {top frame w} \
    .psDg.dimFr.butFr {top frame center} \
    .psDg.dimFr.parFr {top fillx expand}
  
  ####  Buttons frame  ####

  frame .psDg.butFr -borderwidth 0

  set cmd {
    set wfile [.psDg.pathFr.pathEn get]

    set x1 \
      [string trim [.psDg.dimFr.parFr.entryFr.x1En get]]
    set y1 \
      [string trim [.psDg.dimFr.parFr.entryFr.y1En get]]
    set w \
      [string trim [.psDg.dimFr.parFr.entryFr.widthEn get]]
    set h \
      [string trim [.psDg.dimFr.parFr.entryFr.heightEn get]]

    focus .
    grab release .psDg
    destroy .psDg
  }
  append cmd "\n $writeCmd \"\$wfile\" \$x1 \$y1 \$w \$h"
  
  button .psDg.butFr.okBt \
    -text {Ok} \
    -command $cmd \
    -relief raised

  button .psDg.butFr.cancelBt \
    -text {Cancel} \
    -command {
      focus .
      grab release .psDg
      destroy .psDg
    } \
    -relief raised

  #  packing  #
  pack append .psDg.butFr \
    .psDg.butFr.okBt {left fill expand} \
    .psDg.butFr.cancelBt {left fill expand}
  
  ####  Packing toplevel  ####

  pack append .psDg \
    .psDg.titleLb {top frame n fillx pady 10} \
    .psDg.pathFr {top fillx expand pady 16} \
    .psDg.dimFr  {top fillx expand pady 16} \
    .psDg.butFr  {top fillx expand frame s pady 10}

  ####  Entry bindings  ####

  global ioState
  .psDg.pathFr.pathEn insert 0 "$ioState(psFile)"

  bind .psDg.pathFr.pathEn \
    <Return> { focus .psDg.dimFr.parFr.entryFr.x1En }
  bind .psDg.dimFr.parFr.entryFr.x1En \
    <Return> { focus .psDg.dimFr.parFr.entryFr.y1En }
  bind .psDg.dimFr.parFr.entryFr.y1En \
    <Return> { focus .psDg.dimFr.parFr.entryFr.widthEn }
  bind .psDg.dimFr.parFr.entryFr.widthEn \
    <Return> { focus .psDg.dimFr.parFr.entryFr.heightEn }
  bind .psDg.dimFr.parFr.entryFr.heightEn \
    <Return> { focus .psDg.pathFr.pathEn }

  bind .psDg.pathFr.pathEn \
    <Tab> { focus .psDg.dimFr.parFr.entryFr.x1En }
  bind .psDg.dimFr.parFr.entryFr.x1En \
    <Tab> { focus .psDg.dimFr.parFr.entryFr.y1En }
  bind .psDg.dimFr.parFr.entryFr.y1En \
    <Tab> { focus .psDg.dimFr.parFr.entryFr.widthEn }
  bind .psDg.dimFr.parFr.entryFr.widthEn \
    <Tab> { focus .psDg.dimFr.parFr.entryFr.heightEn }
  bind .psDg.dimFr.parFr.entryFr.heightEn \
    <Tab> { focus .psDg.pathFr.pathEn }

  ####  Initial bbox value  ####

  lassign [.f2.f22.cv bbox all] x1 y1 x2 y2
  .psDg.dimFr.parFr.entryFr.x1En insert 0 $x1
  .psDg.dimFr.parFr.entryFr.y1En insert 0 $y1
  .psDg.dimFr.parFr.entryFr.widthEn insert 0 [expr $x2-$x1]
  .psDg.dimFr.parFr.entryFr.heightEn insert 0 [expr $y2-$y1]
}

########  Region Selection  ########

proc iregionStart {X Y} {
  global x1fixed y1fixed
  
  if {[winfo containing $X $Y] != .f2.f22.cv} { return }
  deselect_all
  set x [.f2.f22.cv canvasx [expr $X - [winfo rootx .f2.f22.cv]]]
  set y [.f2.f22.cv canvasy [expr $Y - [winfo rooty .f2.f22.cv]]]

  .f2.f22.cv create rectangle $x $y $x $y \
    -outline red -width 3 -tags {regionSelectFrame}
  set x1fixed 1
  set y1fixed 1
}

proc iregion_move {X Y} {
  set x [.f2.f22.cv canvasx [expr $X - [winfo rootx .f2.f22.cv]]]
  set y [.f2.f22.cv canvasy [expr $Y - [winfo rooty .f2.f22.cv]]]
  dragMotion regionSelectFrame $x $y
}

proc iregionEnd {} {
  global RegionDone RegionX1 RegionY1 RegionWidth RegionHeight

  lassign [.f2.f22.cv coords regionSelectFrame] RegionX1 RegionY1 x2 y2
  .f2.f22.cv delete regionSelectFrame
  set RegionWidth [expr $x2 - $RegionX1]
  set RegionHeight [expr $y2 - $RegionY1]

  

  set RegionDone 1
}

########  Writing Postscript  ########

proc WritePostscript {} {
  if {[llength [.f2.f22.cv find withtag all]] > 0} {
    buildWriteDialog Postscript WritePostscriptInFile
  } else {
    warn "Empty drawing!"
  }
}

proc WritePostscriptInFile {path psx1 psy1 psw psh} {
  global ioState

  set ioState(psFile) "$path"

  set path [sglob "$path"]

  if {$psx1 != ""} {
    waitCursor

    #  Move ctl objects away  #
    set ctlXOffset ""
    if {[.f2.f22.cv find withtag ctl] != ""} {
      set ctlXOffset \
	[expr $psx1 + ($psw) + 50 - ([lindex [.f2.f22.cv bbox ctl] 0]) ]
      .f2.f22.cv move ctl $ctlXOffset 0
    }
puts "Catching..."
    if {[catch {
      .f2.f22.cv postscript -file "$path" \
      -x $psx1 -y $psy1 -width $psw -height $psh
    } error]} {
      MessageBox "Error writing postcript to \n\"$path\" :\n$error"
    }
    
    if {$ctlXOffset != ""} {
      .f2.f22.cv move ctl [expr -($ctlXOffset)] 0
    }
    
    restoreCursor
  } else {
    warn "Empty drawing !"
  }
}

##						     
##  Returns state info, which must say wether the    
##    drawing needs saving
##
## TODO: include image state !
##

proc getCurrentState {} {
  ##  A state is a list of 4 elements : the item ids list, ##
  ##  the types list, the coords list, and the configs list  ##

  #  save current tag #
  set currentItems [.f2.f22.cv find withtag current]
  .f2.f22.cv dtag all current
  
  set ids [.f2.f22.cv find withtag all]
  set hids {}
  set types {}
  set coords {}
  set configs {}
  foreach id $ids {
    lappend hids [list histId $id]
    lappend types [.f2.f22.cv type $id]
    lappend coords [.f2.f22.cv coords $id]
    lappend configs [.f2.f22.cv itemconfigure $id]
  }

  #  restore current tag  #
  foreach it $currentItems {
    .f2.f22.cv addtag current withtag $it
  }

  return [list $hids $types $coords $configs]
}
