# rcsid {$Id: picgeom.tcl,v 4.11 1996/05/21 16:14:58 mangin Rel $}
# This package is free software. Redistribution and use of this file
# are permitted without restrictions.
#
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
#	Frank Mangin
#	Frank.Mangin@ceram.fr  -  Frank.Mangin@sophia.inria.fr

################################################################
####                                                        ####
####      Procedures for interactive geometric              ####
####           transformation of objects                    ####
####                                                        ####
################################################################

proc iopStartHook {idList} {
  foreach id $idList {
    if {[.f2.f22.cv type $id] == "image"} {
      #  image items  #
      .f2.f22.cv addtag laterOp withtag $id
      #      .f2.f22.cv hide $id
      eval ".f2.f22.cv create rectangle [.f2.f22.cv bbox $id] \
			        -width 2 -outline red \
			          -tags {toOp ctl opFr}"
    } else {
      #  other items  #
      set clone \
	[eval .f2.f22.cv create [.f2.f22.cv type $id] \
	 [.f2.f22.cv coords $id] [getSwitches $id]]
      .f2.f22.cv itemconfigure $clone -tags "toOp ctl"
      #      .f2.f22.cv hide $id
    }
  }
}

proc iopEndHook {} {
  .f2.f22.cv delete toOp
  #  .f2.f22.cv show s
}

################################################################
####		   Object duplication                       ####
################################################################

proc Duplicate {} {
  set selection [.f2.f22.cv find withtag s]
  
  if {$selection == ""} {
    warn "No item selected"
    return
  }
  
  foreach item $selection {
    set type   [.f2.f22.cv type $item]
    set coords [.f2.f22.cv coords $item]
    set config [.f2.f22.cv itemconfigure $item]
    set options {}
    foreach cf $config {
      if {[lindex $cf 0] == "-tags"} {
	set tags [lindex $cf 4]
	set newtag "inCreation"
	foreach tag $tags {
	  switch -glob -- $tag {
	    Gr* -
	    tGr* {
	      lappend newtag $tag
	    }
	  }
	}
	lappend options "-tags" $newtag
      } else {
	lappend options [lindex $cf 0] [lindex $cf 4]
      }
    }
    
    set dupid [eval ".f2.f22.cv create $type $coords $options"]
  }
  
  set offset [pgrid 20 20]
  global GridSpacing
  if {[lindex $offset 0] == 0} {
    set offset "$GridSpacing [lindex $offset 1]"
  }
  if {[lindex $offset 1] == 0} {
    set offset "[lindex $offset 0] $GridSpacing"
  } else {
    set offset "[lindex $offset 0] -[lindex $offset 1]"
  }
  eval ".f2.f22.cv move inCreation $offset"
  
  deselect_all
  
  #  Don't call createEndHook because of group tags  #
  updateGroupTags inCreation
  .f2.f22.cv addtag s withtag inCreation
  .f2.f22.cv addtag sOfr withtag inCreation
  .f2.f22.cv dtag all inCreation
  redrawFrames
  global undoCommand
  set undoCommand ".f2.f22.cv delete [.f2.f22.cv find withtag sOfr]"
  canUndo
}

################################################################
####		     Moving objects                         ####
################################################################

proc Move {x y} {
  .f2.f22.cv move sOfr $x $y
  global undoCommand
  set undoCommand ".f2.f22.cv move [.f2.f22.cv find withtag sOfr] [expr \"-($x)\"] [expr \"-($y)\"]"
  canUndo
}

proc imoveSelectionStart {x y} {
  ##  Moves currently selected items, even if  ##
  ##  user didn't click on an item  ##
  global xdragLast ydragLast Dragging

  lassign [pgrid $x $y] x y
  
  ##  First include current in the selection  ##
  if {[select_for_drag] == ""} {
    warn "No item selected"
    deselect_for_drag
  } else {
    dragStartHook
    set xdragLast $x
    set ydragLast $y
  }
  saveCoords s
}

proc imoveStart {x y} {
  ##  Deselect all, and moves item under the mouse  ##
  global xdragLast ydragLast Dragging

  deselect_all
  ##  First select current ##
  if {[select_for_drag] == ""} {
    deselect_for_drag
  } else {
    dragStartHook
    set xdragLast $x
    set ydragLast $y
  }
  saveCoords s
}

proc imoveMotion {x y} {
  global xdragLast ydragLast Dragging
  
  if {!$Dragging} { return }

  lassign [pgrid $x $y] x y

  .f2.f22.cv move sOfr \
    [expr $x - ($xdragLast)] \
    [expr $y - ($ydragLast)]

  set xdragLast $x
  set ydragLast $y
}

proc imoveEnd {} {
  deselect_for_drag
  dragEndHook
  canUndo
}

################################################################
####		    Resizing objects                        ####
################################################################

set xdragOrigin 0
set ydragOrigin 0

set ixdragAnchor 0
set iydragAnchor 0

set dragDox 0
set dragDoy 0

proc iresizeStart {x y} {
  global xdragAnchor ydragAnchor xdragOrigin ydragOrigin dragDox dragDoy
  global ixdragAnchor iydragAnchor

  ##  First include current in the selection  ##

  select_for_drag

  set ids [.f2.f22.cv find withtag s]
  
  if {$ids == ""} {
    warn "No item selected"
    deselect_for_drag
    return
  }
  
  dragStartHook
  lassign [pgrid $x $y] x y
  
  .f2.f22.cv addtag toOp withtag s
  .f2.f22.cv delete sFr
  ## Prepare undo
  saveCoords s
  
  set bb [.f2.f22.cv bbox s]
  
  if {$bb == ""} {
    .f2.f22.cv dtag all toOp
    .f2.f22.cv dtag all laterOp
    deselect_for_drag
    redrawFrames
    dragEndHook
    return
  }

  set x1 [lindex $bb 0]
  set y1 [lindex $bb 1]
  set x2 [lindex $bb 2]
  set y2 [lindex $bb 3]
  
  set attach [cornerOrSide $x $y $bb]
  
  switch -exact $attach {
    none {
      return
    }
    c11 {
      set xdragOrigin $x2
      set ydragOrigin $y2
      set dragDox 1
      set dragDoy 1
    }
    c12 {
      set xdragOrigin $x2
      set ydragOrigin $y1
      set dragDox 1
      set dragDoy 1
    }
    c21 {
      set xdragOrigin $x1
      set ydragOrigin $y2
      set dragDox 1
      set dragDoy 1
    }
    c22 {
      set xdragOrigin $x1
      set ydragOrigin $y1
      set dragDox 1
      set dragDoy 1
    }
    sx1 {
      set xdragOrigin [expr "0.5*($x1+$x2)"]
      set ydragOrigin $y2
      set dragDox 0
      set dragDoy 1
    }
    sx2 {
      set xdragOrigin [expr "0.5*($x1+$x2)"]
      set ydragOrigin $y1
      set dragDox 0
      set dragDoy 1
    }
    sy1 {
      set xdragOrigin $x2
      set ydragOrigin [expr "0.5*($y1+$y2)"]
      set dragDox 1
      set dragDoy 0
    }
    sy2 {
      set xdragOrigin $x1
      set ydragOrigin [expr "0.5*($y1+$y2)"]
      set dragDox 1
      set dragDoy 0
    }
  }
  
  set xdragAnchor [expr "$x - $xdragOrigin"]
  set ixdragAnchor $xdragAnchor
  set ydragAnchor [expr "$y - $ydragOrigin"]
  set iydragAnchor $ydragAnchor
}

proc iresizeMotion {x y} {
  global xdragAnchor ydragAnchor xdragOrigin ydragOrigin dragDox dragDoy
  global Dragging
  
  if {!$Dragging} { return }
  
  lassign [pgrid $x $y] x y

  if { $dragDox } {
    set xfact [expr "($x - $xdragOrigin+0.0)/$xdragAnchor"]
  } else {
    set xfact 1.0
  }
  
  if { $dragDoy } {
    set yfact [expr "($y - $ydragOrigin+0.0)/$ydragAnchor"]
  } else {
    set yfact 1.0
  }
  
  if {($xfact > 0.01) && ($yfact > 0.01)} {
    set xdragAnchor [expr "$x - $xdragOrigin"]
    set ydragAnchor [expr "$y - $ydragOrigin"]
    .f2.f22.cv scale toOp $xdragOrigin $ydragOrigin $xfact $yfact
  }
}

proc iresizeEnd {} {
  global Dragging GriddingMode
  global ixdragAnchor iydragAnchor xdragOrigin ydragOrigin dragDox dragDoy
  
  if {!$Dragging} {
    # never know! #
    dragEndHook
    return
  }

  #  Align objects to grid  #
  if {$GriddingMode} {
    gridObject toOp
  }
  
  ##  Actually resize heavy objects  ##
  waitCursor
  foreach id [.f2.f22.cv find withtag laterOp] {
    .f2.f22.cv dtag $id laterOp
    set fid [.f2.f22.cv find withtag "opSlave$id"]
    if {$fid != ""} {
      eval ".f2.f22.cv coords $id [.f2.f22.cv coords $fid]"
      eval ".f2.f22.cv itemconfigure $id \
              [.f2.f22.cv coords $fid -config]"
    }
  }

  .f2.f22.cv delete opFr
  deselect_for_drag
  .f2.f22.cv dtag all toOp
  .f2.f22.cv addtag s withtag sLt
  .f2.f22.cv dtag all sLt
  redrawFrames
  restoreCursor
  dragEndHook
  canUndo
}

set dragDist 0
set initialDragDist 0

################################################################
####		   Reshaping objects                        ####
################################################################

##
##  Shape states:
##  
##   - ModeState = 0 <==> no handles drawn
##   - ModeState = 1 <==> handles are drawn for ShapeId
##   - ModeState = 2 <==> one of the handles is being dragged.
##  				(and ShapeHid = dragged handle id)
##  

set ShapeId {}
set ShapeHid {}
set ShapeRank {}

##  Create one handle per item coord  ##
proc putShapeHandles {id} {
  global ShapeCoords ShapeHids

  switch [.f2.f22.cv type $id] {
    rectangle -
    image {
      return
    }
    polygon {
      set coords [.f2.f22.cv coords $id]
      set coords [lrange $coords 0 \
		  [expr [llength $coords] - 3]]
    }
    default {
      set coords [.f2.f22.cv coords $id]
    }
  }

  catch {unset ShapeHids}
  catch {unset ShapeCoords}
  for {set i 0} {$i < [llength $coords]} {incr i 2} {

    set x [lindex $coords $i]
    set y [lindex $coords [expr $i + 1]]

    set hid [.f2.f22.cv create rectangle \
	     [expr $x - 3] [expr $y - 3] [expr $x + 3] [expr $y + 3] \
	       -width 2 -outline red \
	       -tags {rsHandle ctl}]
    set ShapeCoords($hid) [list $x $y]
    lappend ShapeHids $hid
  }
}

proc ishapeTake {x y} {
  global ShapeId ShapeCoords ShapeHid ShapeRank 
  global xdragLast ydragLast ModeState

  set id [.f2.f22.cv find withtag current]
  
  if {$id == ""} {
    #  user clicked on nothing  #
    #  -> delete handles  #
    .f2.f22.cv delete rsHandle
    .f2.f22.cv delete opFr
    set ShapeId {}
    set ModeState 0
    return
  }

  if {$ShapeId == ""} {
    #  user clicked on an object, and no handles are drawn #
    #  -> draw handles for object if it supports reshaping #
    set type [.f2.f22.cv type $id]
    if {[lmember {rectangle image text} $type]} {
      warn "Can't reshape $type items"
      return
    }
    .f2.f22.cv delete rsHandle
    .f2.f22.cv delete opFr
    putShapeHandles $id
    set ShapeId $id
    set ModeState 1
    ## Prepare undo
    saveCoords $id
    canUndo
    return
  }

  if {[info exists ShapeCoords($id)]} {
    # handles are drawn, and user clicked on one of them #
    # -> start dragging handle #
    set ModeState 2

    .f2.f22.cv addtag toOp withtag $ShapeId
    
    set ShapeHid $id

    set xdragLast [.f2.f22.cv canvasx $x]
    set ydragLast [.f2.f22.cv canvasy $y]
  } else {
    # handles were drawn, but user clicked  #
    #  on another object -> redraw handles  #
    .f2.f22.cv delete rsHandle
    putShapeHandles $id
    set ShapeId $id
    set ModeState 1
    ## Prepare undo
    saveCoords $id
    canUndo
    return
  }
}

proc ishapeMotion {x y} {
  global xdragLast ydragLast
  global ModeState ShapeId ShapeHid ShapeHids ShapeCoords ShapeRank
  
  if {$ModeState != 2} { return }
  
  lassign [pgrid $x $y] x y
  
  if {($x == $xdragLast) &&
    ($y == $ydragLast)} { return }
  
  set deltax [expr $x - ($xdragLast)]
  set deltay [expr $y - ($ydragLast)]
  #  move handle  #
  .f2.f22.cv move $ShapeHid $deltax $deltay
  
  # update ShapeCoords #
  lassign $ShapeCoords($ShapeHid) xcoord ycoord
  set ShapeCoords($ShapeHid) \
    [list [expr $xcoord + $deltax] [expr $ycoord + $deltay]]
  
  #  reconfigure object  #
  set coords {}
  foreach hid $ShapeHids {
    set coords [concat $coords $ShapeCoords($hid)]
  }

  eval ".f2.f22.cv coords $ShapeId $coords"
  set xdragLast $x
  set ydragLast $y
  update
}

proc ishapeEnd {} {
  global ModeState ShapeHid ShapeId

  if {$ModeState != 2} { return }
  set ModeState 1
  set ShapeHid {}
  .f2.f22.cv dtag all toOp
}

proc ishapeAdd {x y} {
  global ShapeId ShapeCoords ShapeHids ShapeHid

  set coords {}
  set newShapeCoords {}

  if {![lmember {line polygon} [.f2.f22.cv type $ShapeId]]} {
    return
  }
  
  lassign [pgrid $x $y] x y
  set hid [.f2.f22.cv create rectangle \
	   [expr $x - 3] [expr $y - 3] [expr $x + 3] [expr $y + 3] \
	     -width 2 -outline red \
	     -tags {rsHandle ctl}]
  set i [lsearch -exact $ShapeHids $ShapeHid]
  set ShapeHids [linsert $ShapeHids $i $hid]
  set ShapeCoords($hid) [list $x $y]

  eval ".f2.f22.cv coords $ShapeId $coords"
}

proc ishapeDelete {} {
  global ShapeId ShapeCoords ShapeHid ShapeHids

  set coords {}

  switch [.f2.f22.cv type $ShapeId] {
    line {
      if {[llength $ShapeHids] <= 2} { return }
    }
    polygon {
      if {[llength $ShapeHids] <= 3} { return }
    }
    default {
      return
    }
  }
  set i [lsearch $ShapeHids $ShapeHid]
  incr i -1
  if {$i == -1} { set i 1 }
  .f2.f22.cv delete [lindex $ShapeHids $i]
  unset ShapeCoords([lindex $ShapeHids $i])
  set ShapeHids [lreplace $ShapeHids $i $i]

  set coords {}
  foreach hid $ShapeHids {
    set coords [concat $coords $ShapeCoords($hid)]
  }

  eval ".f2.f22.cv coords $ShapeId $coords [lrange $coords 0 1]"

}

################################################################
####		   Rescaling Objects                        ####
################################################################

proc iscaleStart {x y} {
  global dragDist initialDragDist xdragOrigin ydragOrigin

  select_for_drag
  ## Prepare undo
  saveCoords s
  set ids [.f2.f22.cv find withtag s]
  
  if {$ids == ""} {
    warn "No item selected"
    deselect_for_drag
    return
  }
  dragStartHook
  iopStartHook $ids
  lassign [pgrid $x $y] x y
  
  .f2.f22.cv delete sFr
  lassign [.f2.f22.cv bbox s] x1 y1 x2 y2

  set xdragOrigin [expr "0.5 * ($x1 + $x2)"]
  set ydragOrigin [expr "0.5 * ($y1 + $y2)"]
  
  set dragDist \
    [expr "($x-$xdragOrigin)*($x-$xdragOrigin) +\
             ($y-$ydragOrigin)*($y-$ydragOrigin)"]
  set initialDragDist $dragDist
}

proc iscaleMotion {x y} {
  global dragDist xdragOrigin ydragOrigin
  global Dragging
  
  if {!$Dragging} { return }

  lassign [pgrid $x $y] x y
  set dist [expr "($x-$xdragOrigin)*($x-$xdragOrigin) +\
             ($y-$ydragOrigin)*($y-$ydragOrigin)"]
  if {$dist == 0 || $dragDist == 0} { return }
  set fact [expr (0.0 + $dist) / $dragDist]
  
  set dragDist $dist
  .f2.f22.cv scale toOp \
    $xdragOrigin $ydragOrigin $fact $fact
}

proc iscaleEnd {x y} {
  global initialDragDist xdragOrigin ydragOrigin
  global Dragging GriddingMode
  
  if {!$Dragging} {
    dragEndHook
    return
  }

  # compute global scale parameters #
  lassign [pgrid $x $y] x y
  set dist \
    [expr "($x-$xdragOrigin)*($x-$xdragOrigin) +\
             ($y-$ydragOrigin)*($y-$ydragOrigin)"]
  set fact [expr ($dist + 0.0) / $initialDragDist]

  waitCursor
  .f2.f22.cv scale s $xdragOrigin $ydragOrigin $fact $fact
  iopEndHook
  restoreCursor

  deselect_for_drag
  .f2.f22.cv addtag s withtag sLt
  .f2.f22.cv dtag all sLt
  redrawFrames
  dragEndHook
  canUndo
}

##
##  Aligns selected objects according to action
##  Stores object moves into global arrays alignD[xy]
##    so that alignment can be undone (hnone & vnone actions)
##  The arrays are cleared on <Enter> option frame
##

proc alignSelection { action } {
  global alignDx alignDy

  ## Prepare undo
  saveCoords s
  ##  align on the older selected item if we find it  ##
  set bb [.f2.f22.cv bbox firstS]
  if {$bb == ""} {
    if {[set bb [.f2.f22.cv bbox s]] == ""} { cantUndo; return }
  }
  ## find selected groups
  set gtags [getTopGroupTags s]
  
  ## [vh]none actions: just move objects back
  if {$action == "hnone"} {
    foreach gtag $gtags {
      if {[catch {set alignDx($gtag)} dx]} {set dx 0}
      .f2.f22.cv move $gtag [expr "-($dx)"] 0
      set alignDx($gtag) 0
    }
    redrawFrames; canUndo; return
  }
  if {$action == "vnone"} {
    foreach gtag $gtags {
      if {[catch {set alignDy($gtag)} dy]} {set dy 0}
      .f2.f22.cv move $gtag 0 [expr "-($dy)"]
      set alignDy($gtag) 0
    }
    redrawFrames; canUndo; return
  }
  
  ##  set reference positions
  lassign $bb bbx1 bby1 bbx2 bby2
  switch -exact -- $action {
    hcenter { set x0 [expr "0.5 * ($bbx1 + $bbx2)"] }
    hleft   { set x0 $bbx1 }
    hright  { set x0 $bbx2 }
    vcenter { set y0 [expr "0.5 * ($bby1 + $bby2)"] }
    vtop    { set y0 $bby1 }
    vbottom { set y0 $bby2 }
  }

  ## Align each group
  foreach gtag $gtags {
    lassign [.f2.f22.cv bbox $gtag] x1 y1 x2 y2
    set dx 0; set dy 0;
    switch -exact -- $action {
      hcenter { set dx [expr "$x0 - 0.5 * ($x1 + $x2)"] }
      hleft	  { set dx [expr "$x0 - $x1"] }
      hright  { set dx [expr "$x0 - $x2"] }
      vcenter { set dy [expr "$y0 - 0.5 * ($y1 + $y2)"] }
      vtop    { set dy [expr "$y0 - $y1"] }
      vbottom { set dy [expr "$y0 - $y2"] }
    }
    # Update global arrays
    if {[catch {set alignDx($gtag)} olddx]} {
      set olddx 0
    }
    set alignDx($gtag) [expr "$dx + ($olddx)"]
	
    if {[catch {set alignDy($gtag)} olddy]} {
      set olddy 0
    }
    set alignDy($gtag) [expr "$dy + ($olddy)"]
    
    .f2.f22.cv move $gtag $dx $dy
  }
  
  redrawFrames
  canUndo
}

##  Aligning the selection to the grid  ##

proc AlignToGrid {} {

  ## Prepare undo
  saveCoords s
  
  ##  find selected groups  ##
  set gtags [getTopGroupTags s]

  foreach gtag $gtags {
    set ids [.f2.f22.cv find withtag $gtag]
    if {[llength $ids] == 1} {
      set coords [.f2.f22.cv coords $ids]
      if {[llength $coords] == 2} {
	##  Single object with 2 coords (text, ...)  ##
	##  => align coords  ##
	eval ".f2.f22.cv coords $ids \
                [eval "pgrid $coords"]"
	continue
      }
    }

    ##  Grouped objects or multi-coordinate obj.  ##
    ##  => align bbox  ##
    lassign [.f2.f22.cv bbox $gtag] x1 y1
    lassign [pgrid $x1 $y1] X1 Y1
    .f2.f22.cv move $gtag \
      [expr $X1 - $x1] [expr $Y1 - $y1]
  }
  redrawFrames
  canUndo
}
