# rcsid {$Id: picfsbox.tcl,v 4.15 1996/05/21 17:44:22 mangin Rel $}
#
# This package is free software. Redistribution and use of this file
# are permitted without restrictions.
#
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
#	Frank Mangin
#	Frank.Mangin@ceram.fr  -  Frank.Mangin@sophia.inria.fr

##
## Macintosh like File Selection Box
## Version 4.1a3
#
#  Components:
#    - a procedure "fsbox" which pops up a toplevel dialog
#    - a global variable "fsbox" which holds fsbx state information
#  
#  Description:
#    When calling the fsbox proc with non-null okAction or cancelAction,
#      it pops up a dialog and returns immediately. When the users clicks
#      on the ok or cancel button, the corresponding action is invoked,
#      after what the fsbox dialog is destroyed.
#  
#    If called with null okAction and cancelAction, fsbox pops up
#      a dialog, and only returns when the user has clicked on the
#      ok or cancel button. The return value is the absolute selected
#      file path in native format, or "" if the user clicked on cancel
#  
#    $fsbox(dirname) always contains the absolute path name of the
#      currently selected directory in native format, or is ""
#    $fsbox(basename) always contains the base name of the
#      currently selected file ni native format, or is ""
#    $fsbox(action) contains "ok" or "cancel" depending on
#      which button the user clicked on
#
#  Behavior:
#    Macintosh-like, including double-click
#      and jump to first file whose name starts with the typed character
#
#  Syntax:
#    fsbox [typeinflag [message [file [okLabel [okAction [cancelLabel [cancelAction]]]]]]]
#      typein (false):		  true if "1" or "-typein"
#				  if true, causes the dialog to show an editable text entry
#				    allowing the user to type in a non-existing filename
#      message ("Select file:"):  the message to display on top of the fsbox
#      file ("[pwd]"):		  the initial file to select
#  				    (or initial directory to list)
#      okLabel ("Open"):	  the ok button label
#      okACtion (""):		  tcl code to execute when the ok button
#				    is clicked
#      cancelLabel ("Open"):      the cancel button label
#      cancelACtion (""):	  tcl code to execute when the cancel
#				    button is clicked
#  

set fsbox(dirname)   ""
set fsbox(basename)  ""

proc fsbox {{typein 0} {msg "Select file:"} {path ""}
  {okLabel "Open"} {okAction ""}
  {cancelLabel "Cancel"} {cancelAction ""}} {
    global fsbox

    set typein [expr {$typein == 1 || $typein == "-typein"}]
    
    set fsbox(okAction) $okAction
    set fsbox(cancelAction) $cancelAction
    
    if {"$fsbox(dirname)" == ""} { set fsbox(disname) [pwd] }
    if {$path == ""} {
      set path [file join "$fsbox(dirname)" "$fsbox(basename)"]
    }

    if {[isadir "$path"]} {
      set dirname "$path"; set basename ""
    } else {
      set dirname [file dirname "$path"]
      set basename [file tail "$path"]
    }
    
    # make path absolute
    set dirname [file join "[pwd]" "$dirname"]
    
    if {![isadir "$dirname"]} {
      set dirname "$fsbox(dirname)"
      set basename "$fsbox(basename)"
    } else {
      set fsbox(dirname) $dirname
      set fsbox(basename) $basename
    }
      
    set reqname $basename
    if {! [file exists [file join "$fsbox(dirname)" "$fsbox(basename)"]]} {
      set basename ""
    }
    
    ## Build dialog
    catch {destroy .fsbox}
    toplevel .fsbox -borderwidth 0
    wm title .fsbox {File select box}
    wm maxsize .fsbox [winfo screenwidth .] [winfo screenheight .]
    wm minsize .fsbox 100 100
    set x [expr "20 + [winfo rootx .]"]
    set y [expr "20 + [winfo rooty .]"]
    wm geometry .fsbox "+$x+$y"
    
    label .fsbox.mainLb -anchor w \
    -relief flat -text "$msg"
    frame .fsbox.bottomFr -borderwidth 0
    
    frame .fsbox.bottomFr.viewFr -borderwidth 0
    frame .fsbox.bottomFr.ctrlFr -borderwidth 0
    
    menubutton .fsbox.bottomFr.viewFr.dirBt \
    -menu .fsbox.bottomFr.viewFr.dirBt.menu \
    -borderwidth 2
    menu .fsbox.bottomFr.viewFr.dirBt.menu
    
    frame .fsbox.bottomFr.viewFr.scFr \
    -borderwidth 0
    listbox .fsbox.bottomFr.viewFr.scFr.fileLst \
    -selectmode browse \
    -height 16 \
    -yscrollcommand ".fsbox.bottomFr.viewFr.scFr.fileSc set"
    scrollbar .fsbox.bottomFr.viewFr.scFr.fileSc \
    -command ".fsbox.bottomFr.viewFr.scFr.fileLst yview"
    
    button .fsbox.bottomFr.ctrlFr.okBt \
    -text "$okLabel" \
    -command "fsboxOpen"
    
    button .fsbox.bottomFr.ctrlFr.cancelBt \
    -text "$cancelLabel" \
    -command "fsboxCancel"

    if {$typein} {
      entry .fsbox.nameEn -relief sunken
    }
	
    ## Bindings

    bind .fsbox.bottomFr.viewFr.scFr.fileLst \
    <Double-ButtonPress-1> "fsboxUpdateName; fsboxOpen"
    bind .fsbox.bottomFr.viewFr.scFr.fileLst \
    <ButtonPress-1> "+ focus .fsbox.bottomFr.viewFr.scFr.fileLst"
    bind .fsbox.bottomFr.viewFr.scFr.fileLst \
	<Key> "fsboxJump %K"
  
    if {$typein} {
      bind .fsbox.bottomFr.viewFr.scFr.fileLst \
	<Key> "+ fsboxUpdateName;"
      bind .fsbox.bottomFr.viewFr.scFr.fileLst \
	<ButtonRelease-1> fsboxUpdateName
      bind .fsbox.nameEn <Key-KP_Enter> "fsboxUpdateName; fsboxOpen"
      bind .fsbox.nameEn <Key-Return> "fsboxUpdateName; fsboxOpen"      
      #  bind .fsbox.nameEn <Enter> "focus .fsbox.nameEn"
      # bind .fsbox.nameEn <Leave> "focus ."
      bind .fsbox.nameEn <ButtonPress-1> "focus .fsbox.nameEn"
    }      

    ## Packing
    
    pack append .fsbox.bottomFr.ctrlFr \
    .fsbox.bottomFr.ctrlFr.cancelBt {bottom fillx pady 10} \
    .fsbox.bottomFr.ctrlFr.okBt {bottom fillx pady 10}
    
    pack append .fsbox.bottomFr.viewFr.scFr \
    .fsbox.bottomFr.viewFr.scFr.fileLst {left fill} \
    .fsbox.bottomFr.viewFr.scFr.fileSc {left filly}
    
    pack append .fsbox.bottomFr.viewFr \
    .fsbox.bottomFr.viewFr.dirBt {top} \
    .fsbox.bottomFr.viewFr.scFr {top}
    
    pack append .fsbox.bottomFr \
    .fsbox.bottomFr.viewFr {left} \
    .fsbox.bottomFr.ctrlFr {left filly padx 10}
    
    pack append .fsbox \
    .fsbox.mainLb {top pady 5} \
    .fsbox.bottomFr {top}

    if {$typein} {
      pack append .fsbox \
      .fsbox.nameEn {top expand padx 10 pady 10}
      focus .fsbox.nameEn
    } else {
      focus .fsbox.bottomFr.viewFr.scFr.fileLst
    }
    	
    ## Display initial dir/files
    fsboxCd "$fsbox(dirname)" "$fsbox(basename)"
    if {$typein} {
      .fsbox.nameEn delete 0 end
      .fsbox.nameEn insert 0 "$reqname"
      .fsbox.nameEn selection range 0 end 
    }
    
    ## Return values if no actions were specified
    if {"$okAction" == "" && "$cancelAction" == ""} {
      # wait for the box to be destroyed
      update idletask
      #    grab .fsbox
      tkwait window .fsbox
      
      if {"$fsbox(action)" == "ok"} {
        return [file join "$fsbox(dirname)" "$fsbox(basename)"]
      } else {
        return ""
      }
    }
  }
  
##
## Updates the fsbox display
##   and global fsbox variables
## dirname must be an absolute path
##

proc fsboxCd {dirname basename} {
  global fsbox
  
  if {![isadir "$dirname"]} {
    return
  }
  
  set fsbox(dirname) "$dirname"
  set dirComps [file split "$dirname"]
  set cdir [lindex $dirComps end]
  set dirComps [lreplace "$dirComps" end end]
  ## Update current dir button label
  .fsbox.bottomFr.viewFr.dirBt configure \
  -text "$cdir"
  ## Update directory menu entries
  .fsbox.bottomFr.viewFr.dirBt.menu delete 0 end
  for {set i [expr "[llength $dirComps] - 1"]} \
  {$i >= 0} {incr i -1} {
    set path [eval "file join [lrange $dirComps 0 $i]"]
    .fsbox.bottomFr.viewFr.dirBt.menu add command \
    -command "fsboxCd \"$path\" {}" \
    -label "[lindex $dirComps $i]"
  }
  
  ## Update file list content
  .fsbox.bottomFr.viewFr.scFr.fileLst \
  delete 0 end
  set files [fsboxLs $dirname]
  eval ".fsbox.bottomFr.viewFr.scFr.fileLst \
  insert end $files"
  
  if {[isafile [file join "$dirname" "$basename"]]} {
    if {[set i [lsearch -exact $files "$basename"]] >= 0} {
      set fsbox(basename) "$basename"
      .fsbox.bottomFr.viewFr.scFr.fileLst \
      selection set $i $i
      if {[winfo exists .fsbox.nameEn]} {
        .fsbox.nameEn delete 0 end
        .fsbox.nameEn insert 0 "$basename"
      }
    }
  } else {
    set fsbox(basename) ""
    if {[winfo exists .fsbox.nameEn]} {
      .fsbox.nameEn delete 0 end
    }
  }
}

##
##  fsbox dialog ok/cancel button commands
##

proc fsboxOpen {} {
  global fsbox

  if {[winfo exists .fsbox.nameEn]} {
    set current [string trim [.fsbox.nameEn get]]
  } else {
    set i [lindex [.fsbox.bottomFr.viewFr.scFr.fileLst curselection] 0]
    if {$i == ""} { return }
    set current [.fsbox.bottomFr.viewFr.scFr.fileLst get $i]
  }
  
  set path [file join "$fsbox(dirname)" "$current"]

  if {[isadir "$path"]} {
    fsboxCd "$path" ""
  } else {
    set fsbox(basename) "$current"
    if {"$fsbox(okAction)" != ""} {
      eval "$fsbox(okAction)"
    }
    set fsbox(action) ok
    focus .; destroy .fsbox
  }
}

proc fsboxCancel {} {
  global fsbox
  
  if {"$fsbox(cancelAction)" != ""} {
    eval "$fsbox(cancelAction)"
  }
  set fsbox(action) cancel
  focus .; destroy .fsbox
}

##
## Util: returns the list of files in a directory
##   dirname must be the name of a directory
##   the list is sorted (case-sensitive sort -> wrong for the Mac)
## .* is included, but not "." or ".."
##

proc fsboxLs {dirname} {
  if {! [isadir "$dirname"]} {
    return ""
  }
  
  set oldwd [pwd]
  if {[catch {cd "$dirname"}]} {
    cd "$oldwd"
    return
  }
  
  set list [glob -nocomplain -- ".*"]
  set list [concat $list [glob -nocomplain -- "*"]]
  # Remove "." and ".."
  if {[set i [lsearch -exact $list {.}]] >= 0} {
    set list [lreplace $list $i $i]
  }
  if {[set i [lsearch -exact $list {..}]] >= 0} {
    set list [lreplace $list $i $i]
  }
  
  cd "$oldwd"
  return [lsort $list]
}

##
## Util: isfile / isdir
##   follows symlinks
##   false if file doesn't exist
##

proc isadir {path} {
  if {! [file exists "$path"]} {
    return 0
  }
  if {[file isdirectory "$path"]} {
    return 1
  } {
    catch "file type $path" fileType
    if {"$fileType" == "link"} {
      if {[catch {file readlink "$path"} linkName]} {
        return 0
      }
      catch {file type "$linkName"} fileType
      while {"$fileType" == "link"} {
        if {[catch {file readlink "$linkName"} linkName]} {
          return 0
        }
        catch {file type "$linkName"} fileType
      }
      return [file isdirectory "$linkName"]
    }
  }
  return 0
}

proc isafile {path} {
  if {! [file exists "$path"]} {
    return 0
  }
  
  if {[file isfile "$path"]} {
    return 1
  } {
    catch {file type "$path"} fileType
    if {"$fileType" == "link"} {
      if {[catch {file readlink "$path"} linkName]} {
        return 0
      }
      catch {file type "$linkName"} fileType
      while {"$fileType" == "link"} {
        if {[catch {file readlink "$linkName"} linkName]} {
          return 0
        }
        catch {file type "$linkName"} fileType
      }
      return [file isfile "$linkName"]
    }
  }
  return 0
}

##
## fsboxJump
##
## Selects the first file whose first character is
## lexicographically greater than the passwd argument,
##

proc fsboxJump {char} {
  global fsbox

  if {! [regexp -nocase -- {^[a-z]$} $char]} { return }

  ## Deselect current listbox item
  .fsbox.bottomFr.viewFr.scFr.fileLst \
    selection clear 0 end
  set fsbox(basename) {}

  ## Compare to each entry in turn
  set len [.fsbox.bottomFr.viewFr.scFr.fileLst size]
  if {$len == 0} { return }
  incr len -1
  
  for {set i 0} {$i < $len} {incr i} {
    if {[string compare $char \
	 [string range [.fsbox.bottomFr.viewFr.scFr.fileLst get $i] 0 0]] <= 0} {
	   break;
	 }
  }

  ## Don't select any file if char > all, just make last file visible
  if {$i == $len} {
    .fsbox.bottomFr.viewFr.scFr.fileLst see end
    update
    return
  }

  .fsbox.bottomFr.viewFr.scFr.fileLst activate $i
  .fsbox.bottomFr.viewFr.scFr.fileLst selection set $i $i
  .fsbox.bottomFr.viewFr.scFr.fileLst see $i
  update
  set fsbox(basename) [.fsbox.bottomFr.viewFr.scFr.fileLst get $i]
}


##
## Updates the entry's displayed name
##   when the lisbox's selection changes
##

proc fsboxUpdateName {} {
  if {! [winfo exists .fsbox.nameEn]} { return }
  update idletasks
  set i [lindex [.fsbox.bottomFr.viewFr.scFr.fileLst curselection] 0]
  if {$i == ""} { return }
  .fsbox.nameEn delete 0 end
  .fsbox.nameEn insert 0 [.fsbox.bottomFr.viewFr.scFr.fileLst get $i]
  update
}


####################################
##
##  Abort
##  
##  bind .fsbox <Mod1-period> { fsboxAbort }
##

proc fsboxAbort {} {
  global fsbox
  
  set fsbox(action) abort
  catch {destroy .fsbox}
}
