/*
 * TkTreeNode.cc - member routines for class TkTreeNode
 * 
 * Class TkTreeNode redefines virtual functions from its base class,
 * TreeNode, to allow drawing nodes on a tk canvas.
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 */

#include <stdlib.h>
#include <string.h>
#include "TkTreeNode.h"
#include "TkTree.h"
#include "Point.h"


/*
 * construct a node with given tag, pos and size
 */
TkTreeNode::TkTreeNode(TkTree* tree, const char* tag, const char* line, 
    int x, int y, int w, int h, int border)
: TreeNode(tag, x, y, w, h, border), 
  tree_(tree), 
  line_(strdup(line)),
  removeCmd_(NULL)
{
}

/* 
 * destructor
 */
TkTreeNode::~TkTreeNode() {
    if (removeCmd_ != NULL)
	free(removeCmd_);
    free(line_);
}


/*
 * return distance from parent node
 */
int TkTreeNode::parentDistance() const
{
    return tree_->parentDistance();
}
    

/*
 * Return true if the tree's orientation is horizontal
 */
int TkTreeNode::horizontal() const 
{
    return tree_->horizontal();
}

    
/*
 * Draw the tree node using Tk 
 */
void TkTreeNode::drawNode() const
{
    if (tag() != NULL) {
	char buf[64];
	TreePoint p = pos() - prev_pos();
	sprintf(buf, "%d %d", p.x, p.y);
	Tcl_VarEval(tree_->interp(), tree_->canvas(), " move ",
		    "{", tag(), "} ", buf, NULL);
    }
}


/*
 * Remove a tree node using Tk 
 */
void TkTreeNode::removeNode() const
{
    if (tag() != NULL)
	Tcl_VarEval(tree_->interp(), tree_->canvas(), " delete {", tag(), "}", NULL);
    
    if (removeCmd_ != NULL)
        Tcl_Eval(tree_->interp(), removeCmd_);
}


/*
 * Draw the line between this node and its parent 
 */
void TkTreeNode::drawLine(const TreePoint& p1, const TreePoint& p2) const
{
    // make sure the lines are close enough to the nodes in tk 
    // (I'm not sure why this is needed...)
    char buf[80];

    if (horizontal()) 
	sprintf(buf, " %d %d %d %d", p1.x-3, p1.y, p2.x-1, p2.y);
    else 
	sprintf(buf, " %d %d %d %d", p1.x, p1.y-3, p2.x, p2.y-1);

    Tcl_VarEval(tree_->interp(), tree_->canvas(), " coords ", line_, buf, NULL);
}


/*
 * Remove the line from this nodes parent  
 */
void TkTreeNode::removeLine() const
{
    Tcl_VarEval(tree_->interp(), tree_->canvas(), " delete ", line_, NULL);
}


/*
 * This routine is called with the bounding box of the tree before drawing
 * We use this information to change the scrollregion of the tree's canvas
 */
void TkTreeNode::setDrawArea(int x1, int y1, int x2, int y2) const
{
    tree_->setDrawArea(x1, y1, x2, y2);
}


void TkTreeNode::removeCmd(const char* s) {
    if (removeCmd_ != NULL)
	free(removeCmd_);
    removeCmd_ = strdup(s);
}
