# turndial.tcl --
#
# This file defines the default bindings for Tk turndial widgets and provides
# procedures that help in implementing the bindings.
#
# @(#) turndial.tcl 1.06 95/06/22 18:47:27
#
# Copyright (c) 1995 Marco Beijersbergen (beijersb@rulhm1.leidenuniv.nl)
# 
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# Based upon scale.tcl of the tk4.0b4 package:
#   Copyright (c) 1994 The Regents of the University of California.
#   Copyright (c) 1994-1995 Sun Microsystems, Inc.
#
#   See their file "license.terms.tk" for information on usage and
#   redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#

#-------------------------------------------------------------------------
# The code below creates the default class bindings for entries.
#-------------------------------------------------------------------------

# Standard Motif bindings:

bind Turndial <Enter> {
    if $tk_strictMotif {
	set tkPriv(activeBg) [%W cget -activebackground]
	%W config -activebackground [%W cget -background]
    }
    tkTurndialActivate %W %x %y
}
bind Turndial <Motion> {
    tkTurndialActivate %W %x %y
}
bind Turndial <Leave> {
    if $tk_strictMotif {
	%W config -activebackground $tkPriv(activeBg)
    }
    if {[%W cget -state] == "active"} {
	%W configure -state normal
    }
}
bind Turndial <1> {
    tkTurndialButtonDown %W %x %y
}
bind Turndial <B1-Motion> {
    tkTurndialDrag %W %x %y
}
bind Turndial <B1-Leave> { }
bind Turndial <B1-Enter> { }
bind Turndial <ButtonRelease-1> {
    tkCancelRepeat
    tkTurndialEndDrag %W
    tkTurndialActivate %W %x %y
}
bind Turndial <2> {
    tkTurndialButtonDown %W %x %y
}
bind Turndial <B2-Motion> {
    tkTurndialDrag %W %x %y
}
bind Turndial <B2-Leave> { }
bind Turndial <B2-Enter> { }
bind Turndial <ButtonRelease-2> {
    tkCancelRepeat
    tkTurndialEndDrag %W
    tkTurndialActivate %W %x %y
}
bind Turndial <Control-1> {
    tkTurndialControlPress %W %x %y
}
bind Turndial <Up> {
    tkTurndialIncrement %W up little noRepeat
}
bind Turndial <Down> {
    tkTurndialIncrement %W down little noRepeat
}
bind Turndial <Left> {
    tkTurndialIncrement %W up little noRepeat
}
bind Turndial <Right> {
    tkTurndialIncrement %W down little noRepeat
}
bind Turndial <Control-Up> {
    tkTurndialIncrement %W up big noRepeat
}
bind Turndial <Control-Down> {
    tkTurndialIncrement %W down big noRepeat
}
bind Turndial <Control-Left> {
    tkTurndialIncrement %W up big noRepeat
}
bind Turndial <Control-Right> {
    tkTurndialIncrement %W down big noRepeat
}
bind Turndial <Home> {
    %W set [%W cget -from]
}
bind Turndial <End> {
    %W set [%W cget -to]
}

# tkTurndialActivate --
# This procedure is invoked to check a given x-y position in the
# turndial and activate the knob if the x-y position falls within
# the knob.
#
# Arguments:
# w -		The turndial widget.
# x, y -	Mouse coordinates.

proc tkTurndialActivate {w x y} {
    global tkPriv
    if {[$w cget -state] == "disabled"} {
	return;
    }
    if {[$w identify $x $y] == "knob"} {
	if {[$w cget -state] != "active"} {
	    $w configure -state active
	}
    } else {
	if {[$w cget -state] != "normal"} {
	    $w configure -state normal
	}
    }
}

# tkTurndialButtonDown --
# This procedure is invoked when a button is pressed in a turndial.  It
# takes different actions depending on where the button was pressed.
#
# Arguments:
# w -		The turndial widget.
# x, y -	Mouse coordinates of button press.

proc tkTurndialButtonDown {w x y} {
    global tkPriv
    set tkPriv(dragging) 0
    set el [$w identify $x $y]
    if {$el == "left"} {
	tkTurndialIncrement $w up little initial
    } elseif {$el == "right"} {
	tkTurndialIncrement $w down little initial
    } elseif {$el == "knob"} {
	set tkPriv(dragging) 1
	set tkPriv(initValue) [$w get]
	$w set [$w get $x $y]
    }
}

# tkTurndialDrag --
# This procedure is called when the mouse is dragged with
# mouse button 1 down.  If the drag started inside the knob
# (i.e. the turndial is active) then the turndial's value is adjusted
# to reflect the mouse's position.
#
# Arguments:
# w -		The turndial widget.
# x, y -	Mouse coordinates.

proc tkTurndialDrag {w x y} {
    global tkPriv
    if !$tkPriv(dragging) {
	return
    }
    $w set [$w get $x $y]
}

# tkTurndialEndDrag --
# This procedure is called to end an interactive drag of the
# knob.  It just marks the drag as over.
#
# Arguments:
# w -		The turndial widget.

proc tkTurndialEndDrag {w} {
    global tkPriv
    set tkPriv(dragging) 0
}

# tkTurndialIncrement --
# This procedure is invoked to increment the value of a turndial and
# to set up auto-repeating of the action if that is desired.  The
# way the value is incremented depends on the "dir" and "big"
# arguments.
#
# Arguments:
# w -		The turndial widget.
# dir -		"up" means move value towards -from, "down" means
#		move towards -to.
# big -		Size of increments: "big" or "little".
# repeat -	Whether and how to auto-repeat the action:  "noRepeat"
#		means don't auto-repeat, "initial" means this is the
#		first action in an auto-repeat sequence, and "again"
#		means this is the second repetition or later.

proc tkTurndialIncrement {w dir big repeat} {
    global tkPriv
    if {$big == "big"} {
	set inc [$w cget -bigincrement]
	if {$inc == 0} {
	    set inc [expr abs([$w cget -to] - [$w cget -from])/10.0]
	}
	if {$inc < [$w cget -resolution]} {
	    set inc [$w cget -resolution]
	}
    } else {
	set inc [$w cget -resolution]
    }
    if {([$w cget -from] > [$w cget -to]) ^ ($dir == "up")} {
	set inc [expr -$inc]
    }
    $w set [expr [$w get] + $inc]

    if {$repeat == "again"} {
	set tkPriv(afterId) [after [$w cget -repeatinterval] \
		tkTurndialIncrement $w $dir $big again]
    } elseif {$repeat == "initial"} {
	set tkPriv(afterId) [after [$w cget -repeatdelay] \
		tkTurndialIncrement $w $dir $big again]
    }
}

# tkTurndialControlPress --
# This procedure handles button presses that are made with the Control
# key down.  Depending on the mouse position, it adjusts the turndial
# value to one end of the range or the other.
#
# Arguments:
# w -		The turndial widget.
# x, y -	Mouse coordinates where the button was pressed.

proc tkTurndialControlPress {w x y} {
    set el [$w identify $x $y]
    if {$el == "left"} {
	$w set [$w cget -from]
    } elseif {$el == "right"} {
	$w set [$w cget -to]
    }
}
