/*
 * Copyright (c) AT&T Corp. 1994, 1995.
 * This code is licensed by AT&T Corp.  For the
 * terms and conditions of the license, see
 * http://www.research.att.com/orgs/ssr/book/reuse
 */

#include	"dot.h"

#define	NONE	0
#define	NODE	1
#define	EDGE	2

/* MIF font modifiers */
#define REGULAR 0
#define BOLD	1
#define ITALIC	2

/* MIF patterns */
#define P_SOLID	0
#define P_NONE  15
#define P_DOTTED 4	/* i wasn't sure about this */
#define P_DASHED 11 /* or this */

/* MIF bold line constant */
#define WIDTH_NORMAL 1
#define WIDTH_BOLD 3

static	FILE	*Outfile;
static	int		Obj,N_pages;
static 	point	Pages;
static	double	Scale;
static	int		Rot;
static	box		PB;
static int		onetime = TRUE;

typedef struct context_t {
	char 	color_ix,*fontfam,fontopt,font_was_set;
	char	pen,fill,penwidth,style_was_set;
	float	fontsz;
} context_t;

#define MAXNEST 4
static context_t cstk[MAXNEST];
static int SP;

static char *FillStr = "<Fill 3>";
static char *NoFillStr = "<Fill 15>";

static mif_reset()
{
	onetime = TRUE;
}


static init_mif()
{
	SP = 0;
	cstk[0].color_ix = 0;		/* MIF color index 0-7 */
	cstk[0].fontfam = "Times";	/* font family name */
	cstk[0].fontopt = REGULAR;	/* modifier: REGULAR, BOLD or ITALIC */
	cstk[0].pen = P_SOLID;		/* pen pattern style, default is sold */
	cstk[0].fill = P_NONE;
	cstk[0].penwidth = WIDTH_NORMAL;
}

static pointf
mifpt(p)
pointf	p;
{
	pointf	tmp,rv;
	tmp.x = p.x * Scale; tmp.y = Scale * p.y;
	if (Rot == 0) { rv.x = tmp.x; rv.y = PB.UR.y - PB.LL.y - tmp.y; }
	else {rv.x = PB.UR.x - PB.LL.x - tmp.y; rv.y = tmp.x; }
	return rv;
}

static void
mifptarray(A,n)
point		*A;
int			n;
{
	int		i;
	pointf	p;

	fprintf(Outfile," <NumPoints %d>\n",n);
	for (i = 0; i < n; i++) {
		p.x = A[i].x; p.y = A[i].y;
		p = mifpt(p);
		fprintf(Outfile," <Point %.2lf %.2lf>\n",p.x,p.y);
	}
}

static mif_font(cp)
context_t	*cp;
{
	char	*fw,*fa;

	fw = fa = "Regular";
	switch (cp->fontopt) {
		case BOLD: fw = "Bold"; break;
		case ITALIC: fa = "Italic"; break;
	}
	fprintf(Outfile,"<Font <FFamily `%s'> <FSize %.1lf pt> <FWeight %s> <FAngle %s>>\n",cp->fontfam,cp->fontsz,fw,fa);
}

static mif_color(i)
int		i;
{
	fprintf(Outfile,"<Separation %d>\n",i);
}

static mif_style(cp)
context_t	*cp;
{
	fprintf(Outfile,"<Pen %d> <Fill %d> <PenWidth %d>\n",
		cp->pen,cp->fill,cp->penwidth);
}

static mif_comment(obj, sym)
void		*obj;
attrsym_t	*sym;
{
	char	*str;
	str = late_string(obj,sym,"");
	if (str[0]) fprintf(Outfile,"# %s\n",str);
}

static
mif_begin_job(ofp,g,lib,user,vers,pages)
FILE		*ofp;
graph_t		*g;
char		**lib,*user,*vers;
point		pages;
{
	Outfile = ofp;
	Pages = pages;
	N_pages = pages.x * pages.y;
	fprintf(Outfile,"<MIFFile 3.00> # Generated by %s\n",Version);
	fprintf(Outfile,"# For: %s\n",user);
	fprintf(Outfile,"# Title: %s\n",g->name);
	fprintf(Outfile,"# Pages: %d\n",N_pages);
	fprintf(Outfile,"<Units Upt>\n");
}

static 
mif_end_job()
{
	fprintf(Outfile,"# end of MIFFile\n");
}

static
mif_begin_graph(g,bb,pb)
graph_t		*g;
box			bb;				/* drawable region assumed for all pages */
point		pb;				/* device page box (w.r.t. origin) */
{
	g = g;
	PB = bb;
	if (onetime) {
		fprintf(Outfile,"<BRect %d %d %d %d>\n",
			PB.LL.x,PB.UR.y,PB.UR.x - PB.LL.x, PB.UR.y - PB.LL.y);
		init_mif();
		mif_comment(g,agfindattr(g,"comment"));
		onetime = FALSE;
	}
}

static
mif_end_graph()
{
}

static mif_begin_page(page,scale,rot,offset)
point		page;
double		scale;
int			rot;
point		offset;
{
	int		page_number;
	point	sz;

	Scale = scale;
	Rot = rot;
	page_number =  page.x + page.y * Pages.x + 1;
	sz = sub_points(PB.UR,PB.LL);
	fprintf(Outfile," <ArrowStyle <TipAngle 15> <BaseAngle 90> <Length %.1lf> <HeadType Filled>>\n",14*Scale);
}

static mif_end_page()
{
}

static mif_begin_node(n)
node_t		*n;
{
	Obj = NODE;
	fprintf(Outfile,"# %s\n",n->name);
	mif_comment(n,N_comment);
}

static mif_end_node ()
{
	Obj = NONE;
}

static mif_begin_edge (e)
edge_t		*e;
{
	Obj = EDGE;
	fprintf(Outfile,"# %s -> %s\n",e->tail->name,e->head->name);
	mif_comment(e,E_comment);
}

static mif_end_edge ()
{
	Obj = NONE;
}

static mif_begin_context()
{
	assert(SP + 1 < MAXNEST);
	cstk[SP+1] = cstk[SP];
	SP++;
}

static mif_end_context()
{
	int			c, psp = SP - 1;
	assert(SP > 0);
	if (cstk[SP].color_ix != (c = cstk[psp].color_ix)) mif_color(c);
	if (cstk[SP].font_was_set) mif_font(&(cstk[psp]));
	if (cstk[SP].style_was_set) mif_style(&(cstk[psp]));
	/*free(cstk[psp].fontfam);*/
	SP = psp;
}

static mif_set_font(name,size)
char		*name;
double		size;
{
	char	*p,*q,buf[SMALLBUF];
	context_t	*cp;

	cp = &(cstk[SP]);
	cp->font_was_set = TRUE;
	cp->fontsz = Scale * size;
	p = strdup(name);
	if (q = strchr(p,'-')) {
		*q++ = 0;
		canoncolor(q,buf);
		if (streq(buf,"italic")) cp->fontopt = ITALIC;
		else if (streq(q,"bold")) cp->fontopt = BOLD;
	}
	cp->fontfam = p;
	mif_font(&cstk[SP]);
}

static mif_arrowhead(p,theta,scale)
point		p;
double		theta,scale;
{
	pointf		other,mp;
	double		v;
	v = cos(RADIANS(theta)) * ARROW_LENGTH; other.x = v + p.x;
	v = sin(RADIANS(theta)) * ARROW_LENGTH; other.y = v + p.y;
	fprintf(Outfile,"<ArrowStyle <ScaleFactor %.2lf>>\n",scale);
	fprintf(Outfile,"<PolyLine <HeadCap ArrowHead> <NumPoints 2>\n");
		mp = mifpt(other);
		fprintf(Outfile," <Point %.2lf %.2lf>\n",mp.x,mp.y);
		mp.x = p.x; mp.y = p.y;
		mp = mifpt(mp);
		fprintf(Outfile," <Point %.2lf %.2lf>\n",mp.x,mp.y);
	fprintf(Outfile,">\n");
}


static mif_set_color(name)
char	*name;
{
	int		i;
	char	buf[SMALLBUF];
	static char	*mifcolor[]= {"black","white","red","green","blue","cyan",
						"magenta","yellow","comment",(char*)0};
	canoncolor(name,buf);
	for (i = 0; mifcolor[i]; i++)
		if (streq(mifcolor[i],buf))
			{ cstk[SP].color_ix = i; mif_color(i); return; }
	fprintf(stderr,"color %s not supported in MIF\n",name);
}

static mif_set_style(s)
char	**s;
{
	char		*line;
	context_t	*cp;

	cp = &(cstk[SP]);
	while (line = *s++) {
		if (streq(line,"solid")) { /* no-op */ }
		else if (streq(line,"dashed")) cp->pen = P_DASHED;
		else if (streq(line,"dotted")) cp->pen = P_DOTTED;
		else if (streq(line,"invis")) cp->pen = P_NONE;
		else if (streq(line,"bold")) cp->penwidth = WIDTH_BOLD;
		else if (streq(line,"filled")) cp->fill = P_SOLID;
		else if (streq(line,"unfilled")) { /* no-op */ }
		else {
            fprintf(stderr, "mif_set_style: unsupported style %s - ignoring\n",
                line); 
        }
		cp->style_was_set = TRUE;
	}
	if (cp->style_was_set) mif_style(cp);
}

static char *
mif_string(s,auxbuf)
char	*s,*auxbuf;
{
	char			*p = auxbuf,esc;
	while (*s) {
		esc = 0;
		switch (*s) {
			case '\t':	esc = 't'; break;
			case '>': case '\'': case '`': case '\\': esc = *s; break;
		}
		if (esc) {*p++ = '\\'; *p++ = esc;}
		else *p++ = *s;
		s++;
	}
	*p = '\0';
	return auxbuf;
}

static mif_textline(p,str,width,fontsz,align)
point		p;
char		*str;
int			width;
double		fontsz,align;
{
	char	buf[BUFSIZ],*astr;
	pointf	mp;

	mp.x = p.x;
	mp.y = p.y - fontsz/2 + 2;
	mp = mifpt(mp);
	if (align == -.5) astr = "Center";
	else {if (align < 0) astr = "Right"; else astr = "Left";}
	fprintf(Outfile,"<TextLine <Angle %d> <TLOrigin %.2lf %.2lf> <TLAlignment %s>",Rot,mp.x,mp.y,astr);
	fprintf(Outfile," <String `%s'>>\n",mif_string(str,buf));
}

static mif_bezier(A,n)
point		*A;
int			n;
{

	fprintf(Outfile,"<PolyLine <Smoothed Yes> <HeadCap Square>\n");
	mifptarray(A,n);
	fprintf(Outfile,">\n");
}

static mif_polygon(A,n,filled)
point		*A;
int			n,filled;
{
	fprintf(Outfile,"<Polygon %s\n",(filled? FillStr : NoFillStr));
	mifptarray(A,n);
        fprintf(Outfile,">\n");
}

static mif_ellipse(p,rx,ry,filled)
point		p;
int			rx,ry;
int			filled;
{
	pointf		tl,mp;
	tl.x = p.x - rx; tl.y = p.y + ry;
	mp = mifpt(tl);
	fprintf(Outfile,"<Ellipse %s <BRect %.2lf %.2lf %1.lf %1.lf>>\n",
		filled?FillStr:NoFillStr,mp.x,mp.y,Scale*(rx + rx),Scale*(ry + ry));
}

static mif_polyline(A,n)
point		*A;
int			n;
{
	fprintf(Outfile,"<PolyLine <HeadCap Square>\n");
	mifptarray(A,n);
	fprintf(Outfile,">\n");
}

static mif_user_shape(name,A,n,filled)
point		*A;
int			n,filled;
{
	static boolean onetime = TRUE;
	if (onetime) {fprintf(stderr,"custom shapes not available with this driver\n"); onetime = FALSE;}
	mif_polygon(A,n,filled);
}

codegen_t	MIF_CodeGen = {
	mif_reset,
	mif_begin_job, mif_end_job,
	mif_begin_graph, mif_end_graph,
	mif_begin_page, mif_end_page,
	mif_begin_node, mif_end_node,
	mif_begin_edge, mif_end_edge,
	mif_begin_context, mif_end_context,
	mif_set_font, mif_textline,
	mif_set_color, mif_set_style,
	mif_ellipse, mif_polygon,
	mif_bezier, mif_polyline,
	mif_arrowhead,mif_user_shape
};
