# Glyphs
#
# This module offers a simplified interface for accessing the vectorial layouts
# of glyphs contained in OpenType files.
#
# Reference Specification:
#     Opentype 1.6 - http://www.microsoft.com/typography/otspec/
#
# LIMITATIONS
# * TrueTypeCollections are not supported
# * PostScript outlines (OTTO) are not supported.
# * No checksum or full parsing implemented. Just the minumum for extracting glyphs.
# *
#
# CREDITS
#  Thanks to Ricard Marxer for the inspiration  - www.caligraft.com
#  Part of this code is a rework of 
#   * org.apache.batik.svggen project  (Apache License, 2.0)
#   * pdf4tcl project
#     Copyright (c) 2004 by Frank Richter <frichter@truckle.in-chemnitz.de> and
#                       Jens Ponisch <jens@ruessel.in-chemnitz.de>
#     Copyright (c) 2006-2012 by Peter Spjuth <peter.spjuth@gmail.com>
#     Copyright (c) 2009 by Yaroslav Schekin <ladayaroslav@yandex.ru>
#   * sfntutil.tcl - by Lars Hellstrom


 #  Example: see test suite

package require Itcl

 # the following non-standard packages should be installed in 'standard paths'
 # OR within this 'lib' subdir
lappend auto_path [file join [file dirname [file normalize [info script]]] lib]    

package require Bezier
package require BContour

itcl::class Glyphs {
    # --- standard new/destroy -------------------------------  

    proc new {args} {
        set class [namespace current]
         # object should be created in caller's namespace,
         # and fully qualified name should be returned
        uplevel 1 namespace which \[$class #auto $args\]                
    }

    method destroy {} {
        itcl::delete object $this
    }
    # --------------------------------------------------------
    
    private proc string2hex s {
        binary scan $s H* hex
        regsub -all (..) $hex {\\x\1}
    }

    private variable my ; # array used for collecting instance vars ..
    private variable mynameinfo ; # list used for info from name table
    
    constructor {fontPath} {
        set my(fontPath) [file normalize $fontPath]
        
        set my(fd) [open $fontPath "r"]
        fconfigure $my(fd) -translation binary        
        set my(magicTag) [read $my(fd) 4]

        if { $my(magicTag) == "ttcf" } {
           error "ttcf - Not implemented yet"
        }
        if { $my(magicTag) == "OTTO" } {
            error "OpenType: postscript outlines are not supported, yet"
        }        
        if { $my(magicTag) in { "true" "\x00\x01\x00\x00" } } {
           # ... ok
        } else {
            error "Unrecognized magic-number for OpenType font: [string2hex $my(magicTag)]"
        }

        binary scan [read $my(fd) 8] SuSuSuSu numTables searchRange\
            entrySelector rangeShift       
        for {set n 0} {$n<$numTables} {incr n} {
            binary scan [read $my(fd) 16] a4H8IuIu tag checksum start length
            set my(table,$tag) [dict create -checksum $checksum -start $start -length $length]
        }

         # read some tables ...
        foreach tbl {head name hhea maxp hmtx loca cmap} {
            $this _ReadTable.${tbl}
        }
         # .. glyf table, scanned on demand                   
    }

     # from the *numerical* code of a char to its glyph index
     # ( a numerical code is the decimal representation of its (hex) unicode
     #  eg.  946 is the numcode of \u03B2  ( 946=3*256+11*16+2)
    public method numcode2glyphIndex {charCode} {
        set idx 0
        if {  [info exists my(charToGlyph,$charCode)] } {
            set idx $my(charToGlyph,$charCode)
        }
        return $idx    
    }
    
     # from unicode to its glyph index
    public method unicode2glyphIndex {ch} {
        $this numcode2glyphIndex [scan $ch %c]    
    }
    
     # returns a glyph-object for character $ch
     # e.g.
     #     $obj  glyphByUnicode  "A"    ; # simple char
     #     $obj  glyphByUnicode   A     ; # same as above
     #     $obj  glyphByUnicode   ABC   ; # same as above; only the first char is considered !
     #     $obj  glyphByUnicode  \"     ; # the quote
     #     $obj  glyphByUnicode  ""    ; # unicode char \u03B2 (greek letter "Beta")
     #     $obj  glyphByUnicode  \u03B2 ; # same as above
    public method glyphByUnicode {ch} {
        $this glyph [$this unicode2glyphIndex $ch]
    }

     # similar to glyphByUnicode; argument is the *numerical* code of the character         
    public method glyphByCharCode {charCode} {
        $this glyph [$this numcode2glyphIndex $charCode]
    }
    
    private variable glyphCache ; # array : cache of glyphs 
                               # key is glyph-index
                               # value is a glyph-object


     # This is a special method called by the destructor of a single glyph;
     # the glyphs object should release the cache
    private method _forget {glyph} {
          unset glyphCache([$glyph get index])
    }


     # returns the glyph-object stored at $idx position in the glyf-table
     # NOTE: the new glyph-obj will be created in glyphs namespace.
     # NOTE: you *can* destroy single glyph-objs but ut's not necessary; 
     #  all the glyp-obj will be destroyed when the 'main' glyphs-obj is destroyed
    public method glyph {idx} {
        if { ! [info exists glyphCache($idx)] } {      
            set offset [lindex $my(glyphPos) $idx]
             # warning: lindex accepts {} or blancs as index (!) returning the whole list !
             # this should be considered an input error
            if { $offset == {} || ! [string is digit $offset] } {
                error "Glyph $idx - out of index"
            }
            incr offset [dict get $my(table,glyf) -start]
            seek $my(fd) $offset    
            set glyphCache($idx) [Glyph::new $this $idx]
        }
        return $glyphCache($idx)
    }
    
    destructor {
        catch {close $my(fd)}
        foreach {idx obj} [array get glyphCache] {
            $obj destroy
        }       
    }



    # head - Font header table  (required)
    # NOTE: some fields skipped
    #  see http://www.microsoft.com/typography/otspec/head.htm
    private method _ReadTable.head {} {
        set tabInfo $my(table,head)
        seek $my(fd) [dict get $tabInfo -start]        
        set nBytes [dict get $tabInfo -length]
        binary scan [read $my(fd) $nBytes] "SuSuSux6Iu x2 Su x16 S4 x6 SuSu" \
            ver_maj ver_min my(fontRevision) magic \
            my(unitsPerEm) my(bbox) \
            my(indexToLocFormat) my(glyphDataFormat)
        
        if {$ver_maj != 1} {error "Unknown head table version $ver_maj"}
        if {$magic != 0x5F0F3CF5} {error "Invalid head table magic $magic"}
    }

    private common PredefinedNameID ; # array
    array set PredefinedNameID {
      0  {Copyright notice}
      1  {Font Family name}
      2  {Font Subfamily name}
      3  {Unique font identifier}
      4  {Full font name}
      5  {Version string}
      6  {Postscript name}
      7  Trademark
      8  Manufacturer
      9  Designer
      10 Description
      11 {Vendor URL}
      12 {Designer URL}
      13 {License Description}
      14 {License Info URL}
      15 {}
      16 {Preferred Family}
      17 {Preferred Subfamily}
      18 {Compatible Full name}
      19 {Sample text}
      20 {PostScript CID findfont name}
      21 {WWS family name}
      22 {WWS subfamily name}
   }

    # name - Naming table  (required)
    #  see http://www.microsoft.com/typography/otspec/name.htm
    # Really simplified;
    # I don't care to extract all the (repeated) info for different platforms,
    #  encodings, languages.
    # I only extract info for:
    #   platform: 1 (macintosh)
    #   encoding: 0..
    #   language: 0 (American English)
    #
    # info are saved in mynameinfo list - see also nameinfo method
    # each entry of mynameinfo list is a list of {id key value}    
    private method _ReadTable.name {} {
        set tabInfo $my(table,name)
        seek $my(fd) [dict get $tabInfo -start]        
        binary scan [read $my(fd) 6] "SuSuSu"  format count stringOffset
        set storageStart [expr {[dict get $tabInfo -start]+$stringOffset}]
         #Each nameRecord is made of 6 UnsignedShort
        binary scan [read $my(fd) [expr {2*6*$count}]] "Su*"  nameRecord      
        if { $format == 1 } {
            binary scan [read $my(fd) 2] "Su"  langTagCount
             #Each langTagRecord is made of 2 UnsignedShort
            binary scan [read $my(fd) [expr {2*2*$langTagCount}]] "Su"  langTagRecord
             #Language-tag strings stored in the Naming table must be encoded in UTF-16BE.
             #  ... to do ?  don't care ...
        }

         # now decode nameRecords
        set mynameinfo {}
        foreach { platformID encodingID languageID nameID length offset } $nameRecord {
             # Only for platform macintosh, encoding macRoman and language English
            if { $platformID == 1 && $encodingID == 0 && $languageID == 0 } {
                seek $my(fd) [expr {$storageStart+$offset}]
                binary scan [read $my(fd) $length] "a*" value
                ##  NO ! set value1 [encoding convertfrom macRoman $value]

                set nameStr $nameID
                catch {set nameStr $PredefinedNameID($nameID) }
 
                lappend mynameinfo [list $nameID $nameStr $value]
            }
        }
    }    
    
     # each entry of mynameinfo list is a list of {id key value}
     # $fontOBJ nameinfo    ---> list of id, names pairs
     # $fontOBJ nameinfo 44 ---> 3ple  {id name value} about id
     # $fontOBJ nameinfo "Trademark" ---> 3ple  {id name value} about name
     
    public method nameinfo {{what *}} {
        if { $what == "*" } {
            set L {}
            foreach elem $mynameinfo {
                lassign $elem id key
                lappend L $id $key
            }
            return $L
        }
        
        if { [string is digit $what] } {
            set idx [lsearch -index 0 $mynameinfo $what]
            return [lindex $mynameinfo $idx] 
        }
         # else assume what is a key
            set idx [lsearch -index 1 $mynameinfo $what]
            return [lindex $mynameinfo $idx]         
    }

    
    # maxp - Maximum profile table  (required)
    # NOTE: partial parsing; only my(numGlyphs)
    private method _ReadTable.maxp {} {
        set tabInfo $my(table,maxp)
        seek $my(fd) [dict get $tabInfo -start]        
        binary scan [read $my(fd) 6] "SuSuSu" \
            ver_maj ver_min my(numGlyphs)
    }

   
    # hhea - Horizontal Header  (required)
    # - This table contains information for horizontal layout
    # see http://www.microsoft.com/typography/otspec/hhea.htm
    private method _ReadTable.hhea { } {
        set tabInfo $my(table,hhea)
        seek $my(fd) [dict get $tabInfo -start]
        set nBytes [dict get $tabInfo -length]
        binary scan [read $my(fd) $nBytes] "SuSu SSS SuSS SSSS x8 SuSu" \
                ver_maj ver_min \
                my(Ascender) my(Descender) my(LineGap) \
                my(advanceWidthMax) my(minLeftSideBearing) my(minRightSideBearing) \
                my(xMaxExtent) my(caretSlopeRise) my(caretSlopeRun) my(caretOffset) \
                my(metricDataFormat) my(numberOfHMetrics)
        if {$ver_maj != 1} {error "Unknown hhea table version"}
        if {$my(metricDataFormat) != 0} {error "Unknown horizontal metric data format"}
        if {$my(numberOfHMetrics) == 0} {error "Number of horizontal metrics is 0"}   
    }

   # hmtx - Horizontal Metrics  (required)
   # see http://www.microsoft.com/typography/otspec/hmtx.htm
   # NOTE: data from other tables required, 
   #   my(numberOfHMetrics) ...
    private method _ReadTable.hmtx {} {
        set tabInfo $my(table,hmtx)
        seek $my(fd) [dict get $tabInfo -start]
        set nBytes [dict get $tabInfo -length]
        set my(hmetrics) {}       
        for {set glyph 0} {$glyph < $my(numberOfHMetrics)} {incr glyph} {
            # advance width and left side bearing. lsb is actually signed
            # short, but we don't need it anyway (except for subsetting)
            binary scan [read $my(fd) 4] "SuS" aw lsb
            lappend my(hmetrics) [list $aw $lsb]
            if {$glyph == 0} {set my(defaultWidth) $aw}
            if {[info exists my(glyphToChar,$glyph)]} {
                foreach char $my(glyphToChar,$glyph) {
                    set my(charWidths,$char) $aw
                }
            }            
        }
        
        # The rest of the table only lists advance left side bearings.
        # so we reuse aw set by the last iteration of the previous loop.
        # -- BUG (in reportlab) fixed here: aw used scaled in hmetrics,
        # -- i.e. float (must be int)
        for {set glyph $my(numberOfHMetrics)} {$glyph < $my(numGlyphs)} {incr glyph} {
            binary scan [read $my(fd) 2] "Su" lsb
            lappend my(hmetrics) [list $aw $lsb]
            if {[info exists my(glyphToChar,$glyph)]} {
                foreach char $my(glyphToChar,$glyph) {
                    set my(charWidths,$char) $aw
                }
            }
        }
    }       


    # loca - Index to location
    # NOTE: require my(indexToLocFormat)
    # see http://www.microsoft.com/typography/otspec/loca.htm    
    private method _ReadTable.loca {} {
        set tabInfo $my(table,loca)
        seek $my(fd) [dict get $tabInfo -start]
                        
        if {$my(indexToLocFormat) == 0} {
            set nBytes [dict get $tabInfo -length]
            set numGlyphs [expr $nBytes / 2]
            binary scan [read $my(fd) $nBytes] "Su${numGlyphs}" glyphPositions
            foreach el $glyphPositions {
                lappend my(glyphPos) [expr {$el << 1}]
            }
        } elseif {$my(indexToLocFormat) == 1} {
            set nBytes [dict get $tabInfo -length]
            set numGlyphs [expr $nBytes / 4]
            binary scan [read $my(fd) $nBytes] "Iu${numGlyphs}" my(glyphPos)
        } else {
            error "Unknown location table format $my(indexToLocFormat)"
        }
    }

    # cmap - Character to glyph index mapping table
    # NOTE: require ....
    # see http://www.microsoft.com/typography/otspec/cmap.htm        
    private method _ReadTable.cmap {} {
        set tabInfo $my(table,cmap)
        set cmap_offset [dict get $tabInfo -start]
        seek $my(fd) $cmap_offset        
        binary scan [read $my(fd) 4] "SuSu" version cmapTableCount
		set unicode_cmap_offset -1
        for {set f 0} {$f < $cmapTableCount} {incr f} {
            binary scan [read $my(fd) 8] "SuSuIu" platformID encodingID offset
            if {($platformID == 3 && $encodingID == 1) || ($platformID == 0)} {
                # Microsoft, Unicode OR just Unicode
                seek $my(fd) [expr {$cmap_offset+$offset}]
                binary scan [read $my(fd) 2] "Su" format
                if {$format == 4} {
                    set unicode_cmap_offset [expr {$cmap_offset + $offset}]
                    break
                }
            }
            # This SHOULD NOT exit loop:
            if {($platformID == 3 && $encodingID == 0)} {
                seek $my(fd) [expr {$cmap_offset+$offset}]
                binary scan [read $my(fd) 2] "Su" format
                if {$format == 4} {
                    set unicode_cmap_offset [expr {$cmap_offset + $offset}]
                    break
                }
            }
        }
         # we got Format 4
		if { $unicode_cmap_offset == -1 } {
            error "Font does not have cmap for Unicode"
		}
        incr unicode_cmap_offset 2 ; # skip first 2 bytes (format)
        seek $my(fd) ${unicode_cmap_offset}
        binary scan [read $my(fd) 6] "SuSuSu" length language segCount

        set segCount [expr {$segCount / 2}]
        set limit [expr {$unicode_cmap_offset + $length}]
        seek $my(fd) +6 current
        set nBytes [expr 2*${segCount}]        
        binary scan [read $my(fd) $nBytes] "Su${segCount}" endCount
        seek $my(fd) +2 current
        binary scan [read $my(fd) $nBytes] "Su${segCount}" startCount
        binary scan [read $my(fd) $nBytes] "S${segCount}" idDelta
        set idRangeOffset_start [tell $my(fd)]
        binary scan [read $my(fd) $nBytes] "Su${segCount}" idRangeOffset      
        for {set i 0} {$i < $segCount} {incr i} {
            set r_start  [lindex $startCount $i]
            set r_end    [lindex $endCount   $i]
            set r_offset [lindex $idRangeOffset $i]
            set r_delta  [lindex $idDelta $i]           
            for {set uniccode $r_start} {$uniccode <= $r_end} {incr uniccode} {
                if {$r_offset == 0} {
                    set glyph [expr {($uniccode + $r_delta) & 0xFFFF}]
                } else {
                    set offset [expr {($uniccode - $r_start) * 2 + $r_offset}]
                    set offset [expr {$idRangeOffset_start + 2 * $i + $offset}]
                    if {$offset > $limit} {
                        # workaround for broken fonts (like Thryomanes)
                        set glyph 0
                    } else {
                        seek $my(fd) $offset                    
                        binary scan [read $my(fd) 2] "Su" glyph
                        if {$glyph != 0} {
                            set glyph [expr {($glyph + $r_delta) & 0xFFFF}]
                        }
                    }
                }
                set my(charToGlyph,$uniccode) $glyph
                
                ## ?? do we need it ??  see hmtx table ,,,  
                ## once more: do we need it???
                lappend my(glyphToChar,$glyph) $uniccode
            }
        }
    }

    

    private common PublicProps {
        fontPath numGlyphs bbox unitsPerEm fontRevision
        Ascender Descender LineGap advanceWidthMax
        minLeftSideBearing minRightSideBearing xMaxExtent
        caretSlopeRise caretSlopeRun caretOffset
        metricDataFormat numberOfHMetrics    
    }  
    
     # [$obj get] returns all public properties
     # [$obj get $prop] returns the value of the $prop property 
     #   (even it is not public, unsupported)
     # If property does not exist, returns ""
    public method get { {prop {}} } {
        if { $prop == {} } {
             # remove blancs
            return [string trim [regsub -all {\s+} $PublicProps " "]]        
        }
        set res ""
        catch { set res $my($prop) }
        return $res
    }

}


# =============================================================================
# =============================================================================
# =============================================================================


itcl::class Glyph {
    # --- standard new/destroy -------------------------------  

    proc new {args} {
        set class [namespace current]
         # object should be created in caller's namespace,
         # and fully qualified name should be returned
        uplevel 1 namespace which \[$class #auto $args\]                
    }

    method destroy {} {
        itcl::delete object $this
    }
    # --------------------------------------------------------
    
     # convert from 2.14 format (16 bits fixed-decimal format) to double
    private proc f2.14_to_double {x} {
        expr {$x / double(0x4000)}
    } 

     # CONSTANTS
     # ---------
      # Constants for Simple Glyphs
        private common REPEAT_BIT    0x08     
        private common ONCURVE_BIT   0x01        
        private common XSHORTVEC_BIT 0x02
        private common YSHORTVEC_BIT 0x04 
        private common XDUAL_BIT     0x10
        private common YDUAL_BIT     0x20
      # Constants for Composite Glyphs
        private common ARG_1_AND_2_ARE_WORDS  0x0001
        private common ARGS_ARE_XY_VALUES     0x0002
        private common WE_HAVE_A_SCALE        0x0008
        private common WE_HAVE_AN_X_AND_Y_SCALE  0x0040
        private common WE_HAVE_A_TWO_BY_TWO   0x0080
        private common MORE_COMPONENTS        0x0020
        private common WE_HAVE_INSTRUCTIONS   0x0100
                       
    private variable my ; # array used for collecting instance vars ..
    #                        index, bbox, instructions, points


     # NOTE: before creating a new glyph, file-descriptor access postion 
     #   should be set at the beginning of the right 'index' of glyf-table
    constructor { glyphs index } {
        set my(contours) {}
        set my(index) $index ; # just for introspection
        set my(glyphs) $glyphs
        set fd [$glyphs get fd]
        
        binary scan [read $fd 10] "S S4" nOfContours my(bbox)
        set my(instructions) ""
        
        if { $nOfContours >= 0 } {
            $this _SimpleGlyph $fd $nOfContours
        } else {
            $this _CompositeGlyph $fd       
        }
    }
    
    destructor {
        foreach c $my(contours) {
            $c destroy
        }
        catch { $my(glyphs) _forget $this }
    }
    
    
     #side-effect: set 
     #   my(points)       - a list of contours. A contour is a sequence
     #                      { x1 y1 isOn1  x2 y2 isOn2 ... }
     #   my(instructions) ...
    private method _SimpleGlyph {fd nOfContours} {
        set my(instructions) ""
        set my(points) {}
        if { $nOfContours == 0 } return
        
        set nBytes [expr 2*$nOfContours]
        binary scan [read $fd $nBytes] "Su${nOfContours}" endPtsOfContours
        binary scan [read $fd 2] "Su" instructionLength
        set my(instructions) [read $fd $instructionLength]
         # The last end point index reveals the total number of points
        set count [lindex $endPtsOfContours end]
        incr count
        
         # Read the flags array : The flags are run-length encoded
        set flags {}
        for {set i 0} { $i < $count } { incr i } {
            binary scan [read $fd 1] "cu" c
            lappend flags $c
            if { $c &  $REPEAT_BIT } {
                binary scan [read $fd 1] "cu" repeats
                lappend flags {*}[lrepeat $repeats $c]
                incr i $repeats
            }
        }
         # ASSERT:  [llength $flags] == $count
        
        # -- read X coords
        #    The table is stored as relative values, but we'll store them as absolutes        
        # TODO : OPTIMIZE IT- read the whole vector, then parse it ..
        set x 0
        set X {}
        foreach flag $flags {
            if { $flag & $XDUAL_BIT } {
                set rx 0
                if { $flag & $XSHORTVEC_BIT } {
                    binary scan [read $fd 1] "cu" rx
                } 
            } else {
                if { $flag & $XSHORTVEC_BIT } {
                    binary scan [read $fd 1] "cu" rx
                    set rx [expr {-$rx}]                    
                } else {                
                    binary scan [read $fd 2] "S" rx                    
                }
            }
            incr x $rx
            lappend X $x
        }
        # -- read Y coords        
        set y 0
        set Y {}
        foreach flag $flags {
            if { $flag & $YDUAL_BIT } {
                set ry 0
                if { $flag & $YSHORTVEC_BIT } {
                    binary scan [read $fd 1] "cu" ry
                } 
            } else {
                if { $flag & $YSHORTVEC_BIT } {
                    binary scan [read $fd 1] "cu" ry
                    set ry [expr {-$ry}]                    
                } else {                
                    binary scan [read $fd 2] "S" ry                  
                }
            }
            incr y $ry
            lappend Y $y
        }

         # Finally, save X Y and flags in my(points)
         #  grouped by contours ...
        set i 0
        set contours {}
        foreach endPt $endPtsOfContours {
            set contour {}
            while { $i <= $endPt } {
                lappend contour [lindex $X $i] [lindex $Y $i] [expr [lindex $flags $i]& $ONCURVE_BIT]
                incr i
            }
            lappend contours $contour
        }
        set my(points) $contours
    }


    private method _CompositeGlyph { fd } {           
         # first step: simply read and store the list of components (with flags, ...)
        set composite {}
        set flags $MORE_COMPONENTS ; # initial dummy value
        while { $flags & $MORE_COMPONENTS } {        
            binary scan [read $fd 4] "SuSu" flags glyphIndex            
            # Get the arguments as just their raw values
            if { $flags & $ARG_1_AND_2_ARE_WORDS } {
                binary scan [read $fd 4] "SS" argument1 argument2
            } else {
                binary scan [read $fd 2] "cucu" argument1 argument2
            }
            
            set xtranslate 0
            set ytranslate 0
            set point1 0
            set point2 0
            
            set xscale  1.0
            set yscale  1.0
            set scale01 0.0
            set scale10 0.0
                                                                                   
            # Assign the arguments according to the flags
            if { $flags & $ARGS_ARE_XY_VALUES } {
                set xtranslate $argument1
                set ytranslate $argument2
            } else {
                set point1 $argument1
                set pont2 $argument2
            }
            # Get the scale values (if any)
            if {$flags & $WE_HAVE_A_SCALE} {
                # WARNING: it's a 2.14 format ; convert it
                binary scan [read $fd 2] "Su" xscale
                set xscale [f2.14_to_double $xscale]
                set yscale $xscale
            } elseif { $flags & $WE_HAVE_AN_X_AND_Y_SCALE }  {
                binary scan [read $fd 2] "Su" xscale
                set xscale [f2.14_to_double $xscale]
                binary scan [read $fd 2] "Su" yscale
                set yscale [f2.14_to_double $yscale]
            } elseif { $flags & $WE_HAVE_A_TWO_BY_TWO } {
                binary scan [read $fd 2] "Su" xscale
                set xscale [f2.14_to_double $xscale]
                binary scan [read $fd 2] "Su" scale01
                set scale01 [f2.14_to_double $scale01]
                binary scan [read $fd 2] "Su" scale10
                set scale10 [f2.14_to_double $scale10]
                binary scan [read $fd 2] "Su" yscale
                set yscale [f2.14_to_double $yscale]
            }           
            lappend composite [list $glyphIndex $flags [expr $flags & $ARGS_ARE_XY_VALUES] \
               $xtranslate $ytranslate $point1 $point2 $xscale $yscale $scale01 $scale10]               
        }
         #Are there hinting intructions to read?     ( who cares ?)
        if { $flags  & $WE_HAVE_INSTRUCTIONS}  {
            binary scan [read $fd 2] "Su" nInstr
            set my(instructions) [read $fd $nInstr]
        }
        # --------------------------------------------
        # ... and now, ..... lets' glue all parts ...
        # --------------------------------------------
        set glyphsObj $my(glyphs)
        set my(points) {}
        foreach comp $composite {
            lassign $comp glyphIdx flags isXYoffset xtranslate ytranslate point1 point2 xscale yscale scale01 scale10
            if { ! $isXYoffset } {
               error "Sorry, cannot handle weird composite-glyphs !"
            }            
            set contours [[$glyphsObj glyph $glyphIdx] get points]
            foreach contour $contours {
                set newContour {}
                foreach {x y isOnCurve} $contour {
                    set x1 [expr {round($x*$xscale + $y*$scale10) + $xtranslate}]
                    set y1 [expr {round($x*$scale01 + $y*$yscale) + $ytranslate}]
                    lappend newContour  $x1 $y1 $isOnCurve               
                }
                lappend my(points) $newContour                
            }             
        }
    }

 
    public method get { {what {}} }  {
        set validArgs {index bbox points instructions pathlengths paths}    
        if { $what == {} } {
           return $validArgs
        }
        if { $what ni $validArgs } {
           error "wrong arg \"$what\": must be [join $validArgs ", "]"
        }
        switch -- $what {
            paths {              
                if { ! [info exists my(paths)] } {
                    set my(paths) [$this _getSVGpaths]
                }
                return $my(paths)
            }            
            pathlengths {              
                if { ! [info exists my(pathlengths)] } {
                    set my(pathlengths) [$this _getLengths]
                }
                return $my(pathlengths)
            }          
            default {
                 # get the cached value
                return $my($what)            
            }
        }
    }   



    # == auxiliary procs
    private proc midPoint { x0 y0 x1 y1 } {
        list [expr ($x0+$x1)/2.0] [expr ($y0+$y1)/2.0]
    }


     # returns a list of BContours ( a BContour is a sequence of Bezier Curves )   
    private proc _buildContours {gPoints} {
        set bcontours {}
        foreach contour $gPoints {
             # -- Experimental----------------------------------------------------
             # first point on contour MUST be on-curve;
             # if it is not on-curve, 'rotate' the contour (put the first points on tail..)
            set firstOn 0
            foreach {x y isOnCurve} $contour {
                if { $isOnCurve } break
                incr firstOn 3
            }
            if { $firstOn > 0 } {
                set cHead [lrange $contour 0 ${firstOn}-1]
                set contour [lrange $contour $firstOn end]
                lappend contour {*}$cHead
            }
             # -- End of custom exp -----------------------------------------------
            set XYZ {}          
            foreach {x y isOnCurve} $contour {
                lappend XYZ [list $x $y $isOnCurve]
            }
             # close the contour by appending the first point
             # **ONLY** if contour is open !!
            lassign [lindex $XYZ 0] X0 Y0 isOn0
            if { $x != $X0 || $y != $Y0 } {
            	lappend XYZ [lindex $XYZ 0]
            }
        
            set i 0
            set nPoints [llength $XYZ]
            
             # first point is ON the curve
			lassign [lindex $XYZ 0] px0 py0 isOn0
            set bcontour [BContour::new [list $px0 $py0]]
        
            while { $i < $nPoints-1 } {
				lassign [lindex $XYZ $i] px0 py0 isOn0       
                incr i
				lassign [lindex $XYZ $i] px1 py1 isOn1

                if { $isOn0  &&  $isOn1 } {
                     # LINETO
                    $bcontour append [list $px1 $py1]
                    continue
                }
                if { !$isOn0 &&  !$isOn1 } { 
                    lassign [midPoint $px0 $py0 $px1 $py1] mpx mpy
                     # QUADTO
                    $bcontour append [list $px0 $py0] [list $mpx $mpy]
                    continue
                }
                if { !$isOn0 &&  $isOn1 } { 
                     #QUADTO
                    $bcontour append [list $px0 $py0] [list $px1 $py1]
                    continue
                }

                incr i        
				lassign [lindex $XYZ $i] px2 py2 isOn2
    
                if { $isOn0 &&  !$isOn1 && $isOn2 } { 
                     # QUADTO
                    $bcontour append [list $px1 $py1] [list $px2 $py2]
                    continue
                }
                if { $isOn0 &&  !$isOn1 && !$isOn2 } { 
                    lassign [midPoint $px1 $py1 $px2 $py2] mpx mpy
                     # QUADTO
                    $bcontour append [list $px1 $py1] [list $mpx $mpy]
                    continue
                }
                 # no other cases 
            }
            lappend bcontours $bcontour        
        }
        return $bcontours
    }

     # from { {x y} {x y} .... }
     #   to { x y x y ... }
    private proc _flatten { points } {
        set L {}
        foreach P $points {
            lappend L {*}$P
        }
        return $L
    }
	    
     # Returns a list of paths.
     # A path is a list of simple *abstract* commands for drawing lines and
     #  curves (likewise SVG notation).
     # commands are:
     #  MOVETO: M x y    - set the initial point)
     #  LINETO: L x y    - draw a line from current point to (x,y); 
     #                then (x,y) becomes the current point.
     #  QUADTO: Q x1 y1 x2 y2  
     #                - draw a quadratic Bezier curve 
     #                  from current point to (x1,y1) (x2,y2);
     #                  then (x2,y2) becomes the current point.
     #
    private method _getSVGpaths {} {
        set Paths {}
        foreach c [$this _getContours] {
             # get first point of first stroke
            set startPoint [lindex [[$c stroke 0] points] 0]
            set Path {}
            lappend Path [list M {*}$startPoint] ; # MOVETO

            foreach stroke [$c strokes] {
                 # remove the first point (it's equal to the lastof prev stroke)
                set points [lrange [$stroke points] 1 end]
                switch -- [$stroke degree] {
                    0 {
                       # .. ?? valid ??
                    }
                    1 {
                        lappend Path [list L {*}[_flatten $points]] ;# LINETO
                    }
                    2 {
                        lappend Path [list Q {*}[_flatten $points]] ;# QUADTO
                    }
                    default { error "only lines and quadratic bezier expected"}
                }            
            }
            lappend Paths $Path
        }
        return $Paths    
    }
    
    private method _getContours {} {
        if { $my(contours) == {} } {
            set my(contours) [_buildContours $my(points)]
        }
        return $my(contours)    
    }


     # returns a list of N contours-lengths  - N is the number of contours -  
    private method _getLengths {} {
        set L {}
        foreach c [$this _getContours] {
            lappend L [$c length]
        }
        return $L
    }

     # return a list of N list-of-points  (N is the number of contours)
     #  where each point's value is at/tangent_at/normal_at
	 #  or vtangent_at/vnormal_at  (* in these latter 2 cases each returned
	 #  element is not a point; it's a segment (a list of two points))
    public method onUniformDistance { dL meth } {
        set L {}
        foreach c [$this _getContours] {
            lappend L [$c onUniformDistance $dL $meth]
        }
        return $L    
    }
}